<?php
/**
 * Configurator Pattern Model
 *
 * PHP version 5.6
 *
 * @category  ConfiguratorPattern
 * @package   Assets
 * @author    Dibya Jyoti <divya@imprintnext.com>
 * @copyright 2024-2025 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */

 namespace App\Modules\ConfiguratorPatterns\Models;

 use App\Components\Controllers\Component as ParentController;

 /**
 * ConfiguratorPattern
 *
 * @category  ConfiguratorPattern
 * @package   Assets
 * @author    Dibya Jyoti <divya@imprintnext.com>
 * @copyright 2024-2025 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */

 class ConfiguratorPattern extends \Illuminate\Database\Eloquent\Model
{
    protected $table = 'configurator_pattern';
    protected $primaryKey = 'xe_id';
    protected $fillable = ['name', 'price', 'file_name', 'store_id', 'cloud_storage'];
    protected $appends = ['category_names','category_id'];
    public $timestamps = false;

     /**
     * This is a method from Eloquent. The basic functionality of this method is
     * to modify the file_name before sending the response
     *
     * @author divya@imprintnext.com
     * @date   21st March 2024
     * @return file path url
     */
    public function getFileNameAttribute()
    {
        if (isset($this->attributes['file_name']) 
            && $this->attributes['file_name'] != ""
        ) {
            return path('read', 'configurator_pattern') . $this->attributes['file_name'];
        }
        return "";
    }

   /**
     * Create Many-to-Many relationship between Clipart and Category
     *
     * @author divya@imprintnext.com
     * @date   21st March 2024
     * @return relationship object of category
     */
    public function categories()
    {
        return $this->belongsToMany(
            'App\Modules\ConfiguratorPatterns\Models\ConfiguratorPatternCategoryRelationship', 
            'config_pattern_category_rel', 'config_pattern_id', 'category_id'
        );
    }

    public function clipartCategory()
    {
        return $this->hasMany(
            'App\Modules\Cliparts\Models\ClipartCategoryRelation',
            'clipart_id'
        );
    }

      /**
     * Get Category name (comma separated if multiple data is there)
     *
     * @author divya@imprintnext.com
     * @date   22nd March 2024
     * @return relationship object of category
     */
    public function getCategoryNamesAttribute()
    {
        $categoryList = "";
        $parentInit = new ParentController();
        $getData = $parentInit->getCategoriesById(
            'ConfiguratorPatterns', 'ConfiguratorPatternCategoryRelationship', 
            'config_pattern_id', $this->attributes['xe_id'], 
            'name'
        );
        if (!empty($getData) && count($getData) > 0) {
            $categoryList = implode(', ', $getData);
            $categoryList = trim(ltrim(rtrim($categoryList, ','), ','));
        }
        return $categoryList;
    }

     /**
     * Get Category id (comma separated if multiple data is there)
     *
     * @author divya@imprintnext.com
     * @date   22nd March 2024
     * @return relationship object of category
     */
    public function getCategoryIdAttribute()
    {
        $categoryList = "";
        $parentInit = new ParentController();
       
        $getData = $parentInit->getCategoriesById(
            'ConfiguratorPatterns', 'ConfiguratorPatternCategoryRelationship', 
            'config_pattern_id', $this->attributes['xe_id'],  
            'id'
        );
        if (!empty($getData) && count($getData) > 0) {
            $categoryList = implode(', ', $getData);
            $categoryList = trim(ltrim(rtrim($categoryList, ','), ','));
        }
        return $categoryList;
    }
}

