<?php

/**
 * Manage Distresses
 *
 * PHP version 5.6
 *
 * @category  Distress
 * @package   Eloquent
 * @author    Satyabrata <satyabratap@riaxe.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */

namespace App\Modules\Distresses\Controllers;

use App\Components\Controllers\Component as ParentController;
use App\Modules\Distresses\Models\Distress;
use App\Modules\Distresses\Models\DistressCategory as Category;
use Illuminate\Database\Capsule\Manager as DB;

use App\Modules\Distresses\Models\DistressCategoryRelation;
use App\Modules\Distresses\Models\DistressTagRelation;
use App\Modules\PrintProfiles\Models as PrintProfileModels;
use App\Modules\PrintProfiles\Models\PrintProfileAssetsCategoryRel;

/**
 * Distress Controller
 *
 * @category Class
 * @package  Distress
 * @author   Satyabrata <satyabratap@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://inkxe-v10.inkxe.io/xetool/admin
 */
class DistressController extends ParentController
{

    /**
     * POST: Save Distress
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return json response wheather data is saved or any error occured
     */
    public function saveDistresses($request, $response)
    {
        $serverStatusCode = OPERATION_OKAY;
        $success = 0;
        $saveDistressList = [];
        $allFileNames = [];
        $jsonResponse = [
            'status' => 0,
            'message' => message('Distresses', 'error'),
        ];
        // Get Store Specific Details
        $getStoreDetails = get_store_details($request);
        $allPostPutVars = $request->getParsedBody();
        // Save file if request contain files
        $allFileNames = do_upload(
            'upload',
            path('abs', 'distress'),
            [150],
            'array'
        );
        if (empty($allFileNames)) { # do not continue if files are not uploaded
            throw new \Exception('Distress file upload error');
        }
        $uploadingFilesNo = count($allFileNames);
        $allFileNames = array_filter($allFileNames); # to removed empty vlues from array

        $isS3Enabled = $this->checkS3Settings($getStoreDetails['store_id']); # check S3
        $cloudStorage =  ($isS3Enabled ? 1 : 0);
        if (!empty($allFileNames)) {
            foreach ($allFileNames as $eachFile) {
                $lastInsertId = 0;
                $saveDistressList = [];
                if (!empty($eachFile)) {
                    if ($cloudStorage) {
                        $fileToUpload = path('abs', 'distress') . $eachFile;
                        $this->uploadFileToS3("distress", $fileToUpload, $getStoreDetails['store_id']);
                    }
                    $saveDistressList = [
                        'store_id' => $getStoreDetails['store_id'],
                        'name' => $allPostPutVars['name'],
                        'file_name' => $eachFile,
                        'cloud_storage' => $cloudStorage
                    ];
                    $saveEachDistress = new Distress($saveDistressList);
                    if ($saveEachDistress->save()) {
                        $lastInsertId = $saveEachDistress->xe_id;
                        /**
                         * Save category and subcategory data
                         * Category id format: [4,78,3]
                         */
                        if (!empty($allPostPutVars['categories'])) {
                            $categoryIds = $allPostPutVars['categories'];
                            $this->saveDistressCategories(
                                $lastInsertId,
                                $categoryIds
                            );
                        }
                        /**
                         * Save tags
                         * Tag Names format : tag1,tag2,tag3
                         */
                        $tags = !empty($allPostPutVars['tags'])
                            ? $allPostPutVars['tags'] : "";
                        $this->saveDistressTags(
                            $getStoreDetails['store_id'],
                            $lastInsertId,
                            $tags
                        );
                        $success++;
                    }
                }
            }
            $this->memcache('deleteMulti', '', '', "distresses_cat");
            if (!empty($categoryIds)) {
                $catgories = str_replace(array('\'', '"', '[', ']', '<', '>'), ' ', $categoryIds);
                $catIdArr = array_map('trim', array_filter(explode(",", $catgories)));
                foreach ($catIdArr as $categoryId) {
                    $ppAssetCatRelGtInit = new PrintProfileModels\PrintProfileAssetsCategoryRel();
                    $printProfileIds = $ppAssetCatRelGtInit->where('category_id', $categoryId)->pluck('print_profile_id');
                    $this->createUpdateAssetJsonCache($getStoreDetails['store_id'], $printProfileIds, 'distresses', $request, $response);
                }
            }
            if (!empty($success)) {
                $jsonResponse = [
                    'status' => 1,
                    'message' => $success . ' out of ' . $uploadingFilesNo
                        . ' Distress(es) uploaded successfully',
                ];
            }
        }

        // Flush distress memcached
        $this->memcache('deleteMulti', '', '', 'distress');

        return response(
            $response,
            ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

    /**
     * Save Categories/Sub-categories and Distress-Category Relations
     *
     * @param $distressId  Distress ID
     * @param $categoryIds (in  an array with comma separated)
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return boolean
     */
    protected function saveDistressCategories($distressId, $categoryIds)
    {
        $getAllCategoryArr = json_clean_decode($categoryIds, true);
        foreach ($getAllCategoryArr as $catId) {
            $distressCat = new Category();
            $categoriesData = $distressCat->where('xe_id', '=', $catId)->get()->toArray();
            if ($catId == $categoriesData[0]['xe_id']) {
                $flagList = ['asset_available_flag' => 1];
                $distressCat->where('xe_id', '=', $catId)->update($flagList);
            }
            $distressCount = new DistressCategoryRelation();
            $countData =  $distressCount->where('category_id', '=', $catId)->get();
            $countDataArr = $countData->toArray();
            if (count($countData) == 0) {
                if ($catId == $countDataArr[0]['category_id']) {
                    $flagList = ['asset_available_flag' => 0];
                    $distressCat->where('xe_id', '=', $catId)->update($flagList);
                }
            } else {
                if ($catId == $countDataArr[0]['category_id']) {
                    $flagList = ['asset_available_flag' => 1];
                    $distressCat->where('xe_id', '=', $catId)->update($flagList);
                }
            }
        }
        // SYNC Categories to the Distress_Category Relationship Table
        $distressInit = new Distress();
        $findDistress = $distressInit->find($distressId);
        if ($findDistress->categories()->sync($getAllCategoryArr)) {
            return true;
        }
        return false;
    }

    /**
     * Save Tags and Distress-Tag Relations
     *
     * @param $storeId    Store ID
     * @param $distressId Distress ID
     * @param $tags       Multiple Tags
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return boolean
     */
    protected function saveDistressTags($storeId, $distressId, $tags)
    {
        // Save Distress and tags relation
        if (!empty($tags)) {
            $getTagIds = $this->saveTags($storeId, $tags);
            // SYNC Tags into Relationship Table
            $distressInit = new Distress();
            $findDistress = $distressInit->find($distressId);
            if ($findDistress->tags()->sync($getTagIds)) {
                return true;
            }
        } else {
            // Clean relation in case no tags supplied
            $tagRelInit = new DistressTagRelation();
            $distressTags = $tagRelInit->where('distress_id', $distressId);
            if ($distressTags->delete()) {
                return true;
            }
        }
        return false;
    }

    /**
     * GET: List of Distress
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return All/Single Distress List
     */
    public function getDistresses($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $distressData = [];
        $jsonResponse = [
            'status' => 0,
            'data' => [],
            'message' => message('Distress', 'not_found'),
        ];
        // Get Store Specific Details
        $getStoreDetails = get_store_details($request);
        $isS3Enabled = $this->checkS3Settings($getStoreDetails['store_id'], true);
        $s3BaseURL = $isS3Enabled['base_s3_URL'];
        $distressInit = new Distress();
        $getDistresses = $distressInit->where('xe_id', '>', 0)
            ->where('store_id', '=', $getStoreDetails['store_id']);
        $distressId = to_int($args['id']);
        // Total records irrespectable of filters
        $totalCounts = $getDistresses->count();
        if ($totalCounts > 0) {
            if (!empty($distressId)) {
                //For single Distress data
                $distressData = $getDistresses->where('xe_id', '=', $distressId)
                    ->first()->toArray();
                if ($distressData['cloud_storage'] == 1) {
                    $fileName =  $this->getS3URL($distressData['file_name'], $getStoreDetails['store_id']);
                    $thumbFile = $this->getS3URL($distressData['thumbnail'], $getStoreDetails['store_id']);
                    $distressData['file_name'] = $fileName;
                    $distressData['thumbnail'] = $thumbFile;
                }

                $getCategories = $this->getCategoriesById(
                    'Distresses',
                    'DistressCategoryRelation',
                    'distress_id',
                    $distressId
                );
                $getTags = $this->getTagsById(
                    'Distresses',
                    'DistressTagRelation',
                    'distress_id',
                    $distressId
                );
                $distressData['categories'] = $getCategories;
                $distressData['tags'] = $getTags;
                // $distressData = json_clean_decode($distressData, true);
                unset($distressData['category_names']);
                $jsonResponse = [
                    'status' => 1,
                    'data' => [
                        $distressData,
                    ],
                ];
            } else {
                //All Filter columns from url
                $page = $request->getQueryParam('page');
                $perpage = $request->getQueryParam('perpage');
                $categoryId = $request->getQueryParam('category');
                $sortBy = !empty($request->getQueryParam('sortby'))
                    && $request->getQueryParam('sortby') != ""
                    ? $request->getQueryParam('sortby') : 'xe_id';
                $order = !empty($request->getQueryParam('order'))
                    && $request->getQueryParam('order') != ""
                    ? $request->getQueryParam('order') : 'desc';
                $name = $request->getQueryParam('name');
                $printProfileKey = $request->getQueryParam('print_profile_id');
                $type = $request->getQueryParam('type');

                //GET MEMCACHEDATA
                $memcacheData = [];
                if (IS_MEMCACHE and empty($name)) {
                    //MEMCACHED KEY
                    $thisCacheKey = "distress" . "_page" . $page . "_order" . $order . "_perpage" . $perpage . "_s" . $getStoreDetails['store_id'];
                    //Register MEMCACHED KEY in Local Cache
                    $this->createUpdateMemCacheGroup('distress', $thisCacheKey, 'SET');
                    $memcacheData = $this->memcache("get", $thisCacheKey);
                }

                if (empty($memcacheData)) {
                    $offset = 0;
                    // For multiple Distress data
                    $getDistresses->select('xe_id', 'name', 'file_name', 'cloud_storage');
                    // Searching as per name, category name & tag name
                    if (!empty($name)) {
                        $name = '\\' . $name;
                        $getDistresses->where(
                            function ($query) use ($name) {
                                $query->where('name', 'LIKE', '%' . $name . '%')
                                    ->orWhereHas(
                                        'distressTags.tag',
                                        function ($q) use ($name) {
                                            return $q->where(
                                                'name',
                                                'LIKE',
                                                '%' . $name . '%'
                                            );
                                        }
                                    )->orWhereHas(
                                        'distressCategory.category',
                                        function ($q) use ($name) {
                                            return $q->where(
                                                'name',
                                                'LIKE',
                                                '%' . $name . '%'
                                            );
                                        }
                                    );
                            }
                        );
                    }
                    // Filter By Print Profile Id
                    if (!empty($printProfileKey)) {
                        $profileCatRelObj = new \App\Modules\PrintProfiles\Models\PrintProfileAssetsCategoryRel();
                        $assetTypeArr = $this->assetsTypeId('distresses');
                        $profileCatRelDetails = $profileCatRelObj->where(
                            ['asset_type_id' => $assetTypeArr['asset_type_id']]
                        )
                            ->where('print_profile_id', $printProfileKey)
                            ->get();

                        $relCatIds = [];
                        foreach ($profileCatRelDetails->toArray() as $value) {
                            $relCatIds[] = $value['category_id'];
                        }
                        $categoryId = json_clean_encode($relCatIds);
                    }
                    // Filter by Category ID
                    if (!empty($categoryId)) {
                        $searchCategories = json_clean_decode($categoryId, true);
                        $getDistresses->whereHas(
                            'distressCategory',
                            function ($q) use ($searchCategories) {
                                return $q->whereIn('category_id', $searchCategories);
                            }
                        );
                    } else {
                        $searchCategories = [];
                    }
                    if (isset($type) && $type == 'tool') {
                        $getDistresses->whereHas(
                            'distressCategory',
                            function ($q) use ($searchCategories) {
                                return $q->whereIn('category_id', $searchCategories);
                            }
                        );
                    }

                    // Total records including all filters
                    $getTotalPerFilters = $getDistresses->count();
                    // Pagination Data
                    if (!empty($page)) {
                        $totalItem = empty($perpage) ? PAGINATION_MAX_ROW : $perpage;
                        $offset = $totalItem * ($page - 1);
                        $getDistresses->skip($offset)->take($totalItem);
                    }
                    // Sorting All records by column name and sord order parameter
                    if (!empty($sortBy) && !empty($order)) {
                        $getDistresses->orderBy($sortBy, $order);
                    }
                    $currentStoreUrl = '';
                    $databaseStoreInfo = DB::table('stores')->where('xe_id', '=', $getStoreDetails['store_id']);
                    if ($databaseStoreInfo->count() > 0) {
                        $currentStoreUrl = $databaseStoreInfo->get()->first()->store_url;
                    }
                    $distressData = $getDistresses->get()->toArray();

                    foreach ($distressData as &$value) {
                        if ($value['cloud_storage'] == 1) {
                            $value['file_name'] = $value['thumbnail'] = $s3BaseURL . strstr($value['file_name'], 'assets/');
                        } else {
                            $file_name = $value['file_name'];
                            $thumbnail = $value['thumbnail'];
                            $hostname = parse_url($file_name, PHP_URL_HOST); //hostname
                            $value['file_name'] = str_replace($hostname, $currentStoreUrl, $file_name);
                            $value['thumbnail'] = str_replace($hostname, $currentStoreUrl, $thumbnail);
                        }
                    }
                    $jsonResponse = [
                        'status' => 1,
                        'records' => count($distressData),
                        'total_records' => $getTotalPerFilters,
                        'data' => $distressData,
                    ];

                    //SET IN MEMCACHED
                    if (IS_MEMCACHE and empty($name) and isset($thisCacheKey)) {
                        $this->memcache("set", $thisCacheKey, $jsonResponse);
                    }
                } else {
                    $jsonResponse = $memcacheData;
                    if (MEM_MODE) {
                        $jsonResponse['mode'] = 'memcache';
                    }
                }
            }
        }

        return response(
            $response,
            ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

    /**
     * PUT: Update a single distress
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return json response wheather data is updated or not
     */
    public function updateDistress($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Distress', 'not_found'),
        ];
        // Get Store Specific Details from helper
        $getStoreDetails = get_store_details($request);
        $allPostPutVars = $updateData = $request->getParsedBody();
        $categoryIds = (isset($allPostPutVars['categories']) && $allPostPutVars['categories'] != "") ?
            $allPostPutVars['categories'] : NULL;
        $distressId = to_int($args['id']);
        if (!empty($distressId)) {
            $distressInit = new Distress();
            $getOldDistress = $distressInit->where('xe_id', '=', $distressId);
            $distressData = $getOldDistress->get()->toArray();
            $oldCategoryArr = explode(",", $distressData['category_id']);
            $newCategory = str_replace(array('\'', '"', '[', ']', '<', '>'), ' ', $categoryIds);
            $newCategoryArr = explode(",", $newCategory);
            $result = array_diff($oldCategoryArr, $newCategoryArr);
            foreach ($result as $res) {
                $distressCount = new DistressCategoryRelation();
                $countData = $distressCount->where('category_id', '=', $res)->get()->toArray();
                $distressCat = new Category();
                if (count($countData) == 1) {
                    $flagList = ['asset_available_flag' => 0];
                    $distressCat->where('xe_id', '=', $res)->update($flagList);
                } else {
                    $flagList = ['asset_available_flag' => 1];
                    $distressCat->where('xe_id', '=', $res)->update($flagList);
                }
            }
            if ($getOldDistress->count() > 0) {
                unset(
                    $updateData['id'],
                    $updateData['tags'],
                    $updateData['categories'],
                    $updateData['upload'],
                    $updateData['distressId']
                );
                // Delete old file if exist
                $this->deleteOldFile(
                    "distresses",
                    "file_name",
                    [
                        'xe_id' => $distressId,
                    ],
                    path('abs', 'distress')
                );
                $getUploadedFileName = do_upload(
                    'upload',
                    path('abs', 'distress'),
                    [150],
                    'string'
                );
                if (!empty($getUploadedFileName)) {
                    $updateData += ['file_name' => $getUploadedFileName];
                }

                // Update record into the database
                try {

                    $distressInit = new Distress();
                    $distressInit->where('xe_id', '=', $distressId)
                        ->update($updateData);
                    /**
                     * Save category and subcategory data
                     * Category id format: [4,78,3]
                     */
                    if (!empty($allPostPutVars['categories'])) {
                        $categoryIds = $allPostPutVars['categories'];
                        $this->saveDistressCategories($distressId, $categoryIds);
                    }
                    /**
                     * Save tags
                     * Tag Names format : tag1,tag2,tag3
                     */
                    $tags = !empty($allPostPutVars['tags'])
                        ? $allPostPutVars['tags'] : "";
                    $this->saveDistressTags(
                        $getStoreDetails['store_id'],
                        $distressId,
                        $tags
                    );
                    $this->memcache('deleteMulti', '', '', "distresses_cat");
                    if (!empty($categoryIds)) {
                        $catgories = str_replace(array('\'', '"', '[', ']', '<', '>'), ' ', $categoryIds);
                        $catIdArr = array_map('trim', array_filter(explode(",", $catgories)));
                        foreach ($catIdArr as $categoryId) {
                            $ppAssetCatRelGtInit = new PrintProfileModels\PrintProfileAssetsCategoryRel();
                            $printProfileIds = $ppAssetCatRelGtInit->where('category_id', $categoryId)->pluck('print_profile_id');
                            $this->createUpdateAssetJsonCache($getStoreDetails['store_id'], $printProfileIds, 'distresses', $request, $response);
                        }
                    }
                    $jsonResponse = [
                        'status' => 1,
                        'message' => message('Distress', 'updated'),
                    ];
                } catch (\Exception $e) {
                    // Store exception in logs
                    create_log(
                        'Assets',
                        'error',
                        [
                            'message' => $e->getMessage(),
                            'extra' => [
                                'module' => 'Distress',
                            ],
                        ]
                    );
                }
            }
        }
        //Flush distress catagories memcached
        $this->memcache('deleteMulti', '', '', "distresses_cat");
        // Flush distress memcached
        $this->memcache('deleteMulti', '', '', 'distress');

        return response(
            $response,
            ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

    /**
     * DELETE: Delete single/multiple distress
     *
     * @param $request  Slim's Argument parameters
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return json response wheather data is deleted or not
     */
    public function deleteDistress($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Distress', 'not_found'),
        ];
        $getStoreDetails = get_store_details($request);
        if (!empty($args['id'])) {
            $getDeleteIds = $args['id'];
            $getDeleteIdsToArray = json_clean_decode($getDeleteIds, true);
            $totalCount = count($getDeleteIdsToArray);
            if (is_array($getDeleteIdsToArray) && $totalCount > 0) {
                $distressInit = new Distress();
                $getDistress  = $distressInit->whereIn('xe_id', $getDeleteIdsToArray);
                $distressCount = $getDistress->count();
                $singledistress = $getDistress->select('*')->first();
                $distressData = $singledistress->toArray();
                if ($distressCount > 0) {
                    try {
                        $success = 0;
                        foreach ($getDeleteIdsToArray as $distressId) {
                            // Delete from Database
                            $this->deleteOldFile(
                                "distresses",
                                "file_name",
                                [
                                    'xe_id' => $distressId,
                                ],
                                path('abs', 'distress')
                            );
                            //delete file from cloud if enabled
                            if ($distressData['cloud_storage'] == 1) {
                                $this->deleteS3File($distressData['file_name'], $getStoreDetails['store_id']);
                            }
                            // delete file from the server
                            $this->deleteOldFile('distresses', 'file_name', ['xe_id' => $distressId], path('abs', 'distress'), false);
                            $distressDelInit = new Distress();
                            $distressDelInit->where('xe_id', $distressId)->delete();
                            $success++;
                        }
                        if ($success > 0) {
                            $jsonResponse = [
                                'status' => 1,
                                'message' => $success . ' out of ' . $totalCount
                                    . ' Distress(es) deleted successfully',
                            ];
                        }
                    } catch (\Exception $e) {
                        // Store exception in logs
                        create_log(
                            'Assets',
                            'error',
                            [
                                'message' => $e->getMessage(),
                                'extra' => [
                                    'module' => 'Distress',
                                ],
                            ]
                        );
                    }
                }
            }
        }
        // Flush distress memcached
        $this->memcache('deleteMulti', '', '', 'distress');

        return response(
            $response,
            ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

    /**
     * Delete a category from the table
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   20 Jan 2020
     * @return Delete Json Status
     */
    public function deleteCategory($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $getStoreDetails = get_store_details($request);
        $storeId = $getStoreDetails['store_id'];
        $jsonResponse = [
            'status' => 0,
            'message' => message('Category', 'error'),
        ];
        if (!empty($args['id'])) {
            $categoryId = $args['id'];
            $jsonResponse = $this->deleteCat(
                $storeId,
                'distresses',
                $categoryId,
                'Distresses',
                'DistressCategoryRelation'
            );
        }
        $this->memcache('deleteMulti', '', '', "distresses_cat");
        if (!empty($args['id'])) {
            $ppAssetCatRelGtInit = new PrintProfileModels\PrintProfileAssetsCategoryRel();
            $printProfileIds = $ppAssetCatRelGtInit->where('category_id', $args['id'])->pluck('print_profile_id');
            $this->createUpdateAssetJsonCache($getStoreDetails['store_id'], $printProfileIds, "distresses", $request, $response);
        }
        return response(
            $response,
            ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }
}
