<?php

/**
 * Download order details on various endpoints
 *
 * PHP version 5.6
 *
 * @category  Download_Order
 * @package   DownloadOrderArtwork
 * @author    Soumya Swain <soumyas@riaxe.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 */
namespace App\Modules\Orders\Controllers;

use App\Components\Controllers\Component as ParentController;
use App\Dependencies\Zipper as Zipper;
use App\Modules\Orders\Models\OrderItemToken;
use App\Modules\Orders\Controllers\OrderDownloadController;

/**
 * Order Download Controller
 *
 * @category Class
 * @package  DownloadOrderArtwork
 * @author   Soumya Swain <soumyas@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 */
class DownloadArtworkController extends ParentController {
	/**
	 * Integer 96 dpi is default dpi for illustrator
	 */
	public $dpi = 96;

	/**
	 * String The path to the current order zip file
	 */
	public $orderPath;

	/**
	 * String The path to the current order zip file
	 */
	public $sidePath;

	/**
	 * String The path to the current svg save file
	 */
	public $svgSavePath;

	/**
	 * Array empty print color array
	 */
	public $printColorsArr = array();

	/**
	 * String svg image tag
	 */
	public $productImageTag;

	/**
	 * String print unit for current order
	 */
	public $printUnit;

	/**
	 * Html dom object
	 */
	public $domObj;

	/**
	 * Initial filter variable
	 */
	public $filter;

	/**
	 * Array SVG file format enabled print method
	 * Remove latter
	 */
	public $svgFileArr = array();

	/**
	 * Define order path
	 **/
	public function __construct() {
		$this->createOrderArtworkDirectory(ASSETS_PATH_W . 'orders_artwork');
		$this->orderPath = path('abs', 'orders_artwork');
		$this->domHtmlPathInclue();
	}
	/**
	 * create order artwork directory
	 **/
	public function createOrderArtworkDirectory($dir) {
		if (!is_dir($dir)) {
			mkdir($dir, 0755, true);
		}
	}
	/**
	 * GET : Download vendor artwork
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author soumyas@riaxe.com
	 * @date   04 June 2020
	 * @return json response wheather data is updated or not
	 */
	public function downloadOrderArtwork($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$storeDetails = get_store_details($request);
        $storeID = $storeDetails['store_id'] ? $storeDetails['store_id'] : 1;
        $isS3Enabled = $this->checkS3Settings($storeID);
		$status = false;
		$this->orderPath = path('abs', 'order');
		if (isset($args) && !empty($args)) {
			$orderSrcPath = path('abs', 'order');
			$token = $args['token'];
			$tokenData = $this->explodeToken($token);
			$orderItemTokenInit = new OrderItemToken();
			$orderId = $tokenData['orderId'];
			$orderItemId = $tokenData['orderItemId'];
			$tokenReturn = $orderItemTokenInit->select('token')
				->where('order_id', $orderId)
				->where('order_item_id', $orderItemId)
				->get()->toArray();
			//For Shopify order data is saved in order number so changed the folder name to order number
			//This is done with a purpose. do not change it. 
			$orderId = (strtolower(STORE_NAME) == "shopify")?$tokenData['orderNumber']:$tokenData['orderId'];
			if (!empty($tokenReturn[0])) {
				if (!is_dir($this->orderPath . $orderId)) {
					mkdir($this->orderPath . $orderId, 0755, true);
				}
				$orderPathOriginal = $orderSrcPath . $orderId . '/';
				$orderDestPath = $this->orderPath . $orderId;
				/* copy files */
				$this->recurse_copy($orderPathOriginal, $orderDestPath);

				$downloadZipPath = $this->orderPath . "order_" . $orderId . "_item_" . $orderItemId . ".zip";
				$orderDownloadObj = new OrderDownloadController();
				$status = $orderDownloadObj->downloadOrderByItemId($orderId, $orderItemId , $storeID, $isS3Enabled);
				if ($status) {
					$returnStatus = $orderDownloadObj->createOrderZipFileByItemId($request, $response, $orderId, $orderItemId, $orderId, $storeID, $isS3Enabled);
					if ($returnStatus) {
						$status = true;
						$orderDownloadObj->zipFileDownload($downloadZipPath);
					}
				}
				$msg = $status ? 'done' : 'error';
				$jsonResponse = [
					'status' => $status,
					'message' => message('order download', $msg),
				];
			} else {
				$jsonResponse = [
					'status' => 0,
					'message' => 'Invalid token',
				];
			}
		} else {
			$jsonResponse = [
				'status' => 0,
				'message' => 'token empty',
			];
		}
		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}
	/**
	 *
	 * Explode the Token and return in array
	 *
	 * @author soumyas@riaxe.com
	 * @date   4 June 2020
	 * @return array
	 *
	 */
	public function explodeToken($token) {
		$result = array();
		$token = base64_decode($token);
		$values = explode("&", $token);
		foreach ($values as $value) {
			$resArray = explode("=", $value);
			$result[$resArray[0]] = $resArray[1];
		}
		return $result;
	}


	/**
	 * GET: include dom html file
	 *
	 * @author radhanatham@riaxe.com
	 * @date   04 April 2020
	 * @return nothing
	 */
	private function domHtmlPathInclue() {
		include_once dirname(__FILE__) . '/../../../Dependencies/simple_html_dom.php';
	}
	/**
	 * Delete: Directory
	 *
	 * @author soumyas@riaxe.com
	 * @date   05 June 2020
	 * @return nothing
	 */
	public function deleteDirectory($dirname) {
		if (is_dir($dirname)) {
			$dir_handle = opendir($dirname);
		}
		if (!$dir_handle) {
			return false;
		}
		while ($file = readdir($dir_handle)) {
			if ($file != "." && $file != "..") {
				if (!is_dir($dirname . "/" . $file)) {
					unlink($dirname . "/" . $file);
				} else {
					delete_directory($dirname . '/' . $file);
				}

			}
		}
		closedir($dir_handle);
		rmdir($dirname);
		return true;
	}
	/**
	 *
	 * Delete token
	 * @param $orderId
	 * @param $orderItemId
	 * @author soumyas@riaxe.com
	 * @date   04 June 2020
	 * @return array
	 *
	 */
	public function deleteOrderToken($orderId, $orderItemId) {
		$tokenInit = new OrderItemToken();
		$tokenDelete = $tokenInit->where(
			['order_id' => $orderId, 'order_item_id' => $orderItemId]
		);
		return $tokenDelete->delete();
	}
	/**
	 *
	 * Recurse copy
	 * @param $src
	 * @param $dst
	 * @author soumyas@riaxe.com
	 * @date   04 June 2020
	 * @return true
	 *
	 */
	protected function recurse_copy($src, $dst) {
		$dir = opendir($src);
		@mkdir($dst);
		while (false !== ($file = readdir($dir))) {
			if (($file != '.') && ($file != '..')) {
				if (is_dir($src . '/' . $file)) {
					$this->recurse_copy($src . '/' . $file, $dst . '/' . $file);
				} else {
					@copy($src . '/' . $file, $dst . '/' . $file);
				}
			}
		}
		closedir($dir);
	}
}