<?php
/**
 * Manage Production
 *
 * PHP version 5.6
 *
 * @category  Productions
 * @package   Production_Hub
 * @author    Debashri Bhakat <debashrib@riaxe.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 */

namespace App\Modules\Productions\Controllers;

use App\Components\Controllers\Component as ParentController;
use App\Components\Models\ProductionAbbriviations;
use App\Components\Models\PurchaseOrderStatus as PurchaseOrderStatus;
use App\Modules\Orders\Controllers\OrdersController;
use App\Modules\PrintProfiles\Models\PrintProfile;
use App\Modules\Productions\Models\Orders;
use App\Modules\Productions\Models\ProductionEmailTemplates;
use App\Modules\Productions\Models\ProductionHubSetting;
use App\Modules\Productions\Models\ProductionJobAgents;
use App\Modules\Productions\Models\ProductionJobLog;
use App\Modules\Productions\Models\ProductionJobNoteFiles;
use App\Modules\Productions\Models\ProductionJobNotes;
use App\Modules\Productions\Models\ProductionJobs;
use App\Modules\Productions\Models\ProductionJobStages;
use App\Modules\Productions\Models\ProductionStatus;
use App\Modules\Productions\Models\PurchaseOrderItems;
use App\Modules\Productions\Models\StatusAssigneeRel;
use App\Modules\Productions\Models\StatusFeatures;
use App\Modules\Productions\Models\StatusPrintProfileRel;
use App\Modules\Productions\Models\User;
use App\Modules\Productions\Models\UserRole;
use App\Modules\Productions\Models\UserRoleRel;
use Illuminate\Database\Capsule\Manager as DB;
use App\Modules\Customers\Controllers\CustomersController;
use App\Modules\Productions\Models\ProductionJobHolidays;
use App\Modules\Products\Controllers\ProductsController;
use App\Modules\BarCode\Models\Barcodes;
use App\Modules\Settings\Models\Language;

/**
 * Production Controller
 *
 * @category Productions
 * @package  Production_Hub
 * @author   Debashri Bhakat <debashrib@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 */
class ProductionController extends ParentController {

	/**
	 * Delete : Delete Status from the table if not
	 * associate with quote
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   24 Sept 2020
	 * @return Delete Json Status
	 */
	public function deleteProductionStatus($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Status', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$storeId = $request->getQueryParam('store_id') ? $request->getQueryParam('store_id') : 1;
		if (!empty($args) && $args['id'] > 0) {
			$statusId = to_int($args['id']);
			$module = $request->getQueryParam('module') ? $request->getQueryParam('module') : '';
			if ($module == 'po') {
				$purchaseOrderStatusInit = new PurchaseOrderStatus();
				$purchaseOrderStatusArr = $purchaseOrderStatusInit->select('xe_id')
					->where(
						[
							'xe_id' => $statusId,
							'store_id' => $getStoreDetails['store_id'],
						]
					);
				if ($purchaseOrderStatusArr->count() > 0) {
					$xeId = $purchaseOrderStatusArr->get()->toArray();
					$purchaseOrderStatusInit->where(['xe_id' => $xeId[0]['xe_id']])->delete();
					$jsonResponse = [
						'status' => 1,
						'message' => message('Purchase order status', 'deleted'),
					];
				}

			} else {
				$statusInit = new ProductionStatus();
				$status = $statusInit->find($statusId);
				$oldStatusName = $status->status_name;
				$oldStatusSlug = $status->slug;
				if (isset($status->xe_id) && $status->xe_id != "" && $status->xe_id > 0) {
					if ($status->delete()) {
						//Delete Email template associated with status
						$oldTemplateTypeName = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '_', $oldStatusName)));
						$templateInit = new ProductionEmailTemplates();
						$tempDataArr = $templateInit->select('xe_id')
							->where(
								[
									'template_type_name' => $oldTemplateTypeName,
									'module_id' => 4,
									'store_id' => $getStoreDetails['store_id'],
								]
							);
						if ($tempDataArr->count() > 0) {
							$tempData = $tempDataArr->get();
							$tempData = json_clean_decode($tempData, true);
							$templateTypeId = $tempData[0]['xe_id'];
							$emailTemp = $templateInit->find($templateTypeId);
							$emailTemp->delete();
						}
						//Delete data from production_status_print_profile_rel table
						$statusPrintProfileRelInit = new StatusPrintProfileRel();
						$statusPrintProfileRelInit->where(['status_id' => $statusId])->delete();

						//Delete data from production_status_features table
						$statusFeaturesInit = new StatusFeatures();
						$statusFeaturesInit->where(['status_id' => $statusId])->delete();

						//Delete data from production_status_assignee_rel table
						$statusAssigneeRelInit = new StatusAssigneeRel();
						$statusAssigneeRelInit->where(['status_id' => $statusId])->delete();

						//Delete data from barcode table
						$barCodesInit = new BarCodes();
						$getBarCodeData = $barCodesInit->where(['slug' => $oldStatusSlug , 'store_id' => $storeId , 'module_id' => 4]);
						if($getBarCodeData->count() > 0) {
							$getBarCodeArray =  $getBarCodeData->first()->toArray();
							$barcodeImage =  $getBarCodeArray['barcode_image'];
							$barCodeImagePath =  ASSETS_PATH_W.'bar_codes/'.$barcodeImage;
							$getBarCodeData->delete();
							unlink($barCodeImagePath);
						}

					}
					$jsonResponse = [
						'status' => 1,
						'message' => message('Production Status', 'deleted'),
					];
				}
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Generate Job id
	 *
	 * @param $request  Slim's Request object
	 *
	 * @author debashrib@riaxe.com
	 * @date   25 Sept 2020
	 * @return json response
	 */

	private function generateJobId($request, $lastJobId = '') {
		//Get quotation setting data
		$getStoreDetails = get_store_details($request);
		$settingInit = new ProductionHubSetting();
		$settingData = $settingInit->select('setting_value', 'flag')
			->where([
				'module_id' => 4,
				'setting_key' => 'job_card',
				'store_id' => $getStoreDetails['store_id'],
			]);
		if ($settingData->count() > 0) {
			$settingDataArr = $settingData->first()->toArray();
			$settingValue = json_clean_decode($settingDataArr['setting_value'], true);
			$preFix = isset($settingValue['prefix']) ? $settingValue['prefix'] : '';
			$startingNum = isset($settingValue['starting_number']) ? $settingValue['starting_number'] : '';
			$postFix = isset($settingValue['postfix']) ? $settingValue['postfix'] : '';
			$flag = 0;
			if ($settingDataArr['flag'] == 1 && $flag == 1) {
				$flag = 1;
				$newJobId = $preFix . $startingNum . $postFix;
			} elseif ($lastJobId == '') {
				$newJobId = $preFix . $startingNum . $postFix;
			} else {
				$postFixLen = strlen($postFix);
				$preFixLen = strlen($preFix);
				if ($preFixLen > 0) {
					if(0 === strpos($lastJobId, $preFix)){
	                    $withoutPrefix = substr($lastJobId, strlen($preFix)).'';
	                }
				} else {
					$withoutPrefix = $lastJobId;
				}
                if ($postFixLen > 0) {
                	$jobNum = substr($withoutPrefix, 0, -$postFixLen);
				} else {
					$jobNum = $withoutPrefix;
				}
				$newJobNum = $jobNum + 1;
				$newJobId = $preFix . $newJobNum . $postFix;
			}
			$productionJobInit = new ProductionJobs();
			$jobData = $productionJobInit->where(
				[
					'job_id' => $newJobId,
				]);
			if ($jobData->count() > 0) {
				return $this->generateJobId($request, $newJobId);
			} else {
				return $newJobId;
			}
		}
	}

	/**
	 * POST: Create Production Job
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   25 Sept 2020
	 * @return json response
	 */
	public function createProductionJob($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$productionJobInit = new ProductionJobs();
		$orderInit = new Orders();
		$allPostPutVars = $request->getParsedBody();
		$orderIdArr = $allPostPutVars['order_id'];
		$orderIdArr = json_clean_decode($orderIdArr, true);
		if (isset($orderIdArr) && !empty($orderIdArr)) {
			//check for agent assignment in all stages
			$settingData = $this->getProductionSetting($request, $response, ['module_id' => 4, 'return_type' => 1]);
			$settingData = $settingData['data'];
			$statusDataArr = $settingData['status'];
			$isAgentAssign = true;
			foreach ($statusDataArr as $stage) {
				if (empty($stage['assignee_id'])) {
					$isAgentAssign = false;
				}
			}
			if ($isAgentAssign) {
				$finalSendingEmailData = [];
				foreach ($orderIdArr as $orderId) {
					$ordersControllerInit = new OrdersController();
	            	$orderDetails = $ordersControllerInit->getOrderDetails($request, $response, ['id' => $orderId], 1);
					$orderDetails = $orderDetails['data'];
					$customerId = $orderDetails['customer_id'];
					$orderItems = $orderDetails['orders'];
					$checkDuplicateOrder = $productionJobInit->where('order_id', $orderId);
					//Check if production job is created for this order or not
					if ($checkDuplicateOrder->count() == 0) {
						//Change the Order Production Status to In-Progress
						$checkOrderProductionJob = $orderInit->where([
							'order_id' => $orderId,
							'production_status' => 0,
						]);
						//Check if production job is started
						if ($checkOrderProductionJob->count() == 1) {
							$lastJobId = '';
							$allProductArr = [];
							foreach ($orderItems as $items) {
								//check for product
								if (!in_array($items['product_id'], $allProductArr)) {
								    array_push($allProductArr, $items['product_id']);
								}
								//Generate Production Job Id
								$lastRecord = $productionJobInit->select('job_id')->latest()->first();
								if (!empty($lastRecord)) {
									$lastJobId = $lastRecord->job_id;
								}
								$jobId = $this->generateJobId($request, $lastJobId);
								$postData = [
									'store_id' => $getStoreDetails['store_id'],
									'job_id' => $jobId,
									'order_id' => $orderId,
									'order_item_id' => $items['id'],
									'order_item_quantity' => $items['quantity'],
									'job_title' => ($items['sku'] != '') ? $items['name'] . ':' . $items['sku'] : $items['name'],
									'job_status' => 'progressing',
									'current_stage_id' => 0,
									'created_at' => date_time(
										'today', [], 'string'
									)
								];
								$productionJob = new ProductionJobs($postData);
								if ($productionJob->save()) {
									$productionJobLastId = $productionJob->xe_id;
									//Change the production setting flag value after production job is created
	                				$this->changeSettingFlagValue($getStoreDetails['store_id'], 4, 'job_card'); 
									//Adding to production job log for job creation
									$logData = [
										'job_id' => $productionJobLastId,
										'title' => 'Job created',
										'description' => 'New job #' . $jobId . ' created.',
										'user_type' => $allPostPutVars['user_type'],
										'user_id' => $allPostPutVars['user_id'],
										'created_date' => date_time(
											'today', [], 'string'
										)
									];
									$this->addingProductionLog($logData);

									$decorationSettingsData = $items['decoration_settings_data'];
									$localStageFlag = 0;
									if (!empty($decorationSettingsData)) {
										//Get all the print method associated with this item
										$printMethodArr = $this->getPrintMethodOfOrder($items);
										$startingDate = date_time(
											'today', [], 'string'
										);
										foreach ($printMethodArr as $printKey => $method) {
											//Get associated stage
											$stageArr = $this->getStagesWrtPrintMethod($method['print_method_id']);
											foreach ($stageArr as $stageKey => $stages) {
												$productionJobStageInit = new ProductionJobStages();
												$checkStageData = $productionJobStageInit->where('job_id', $productionJobLastId);
												$checkStageDataCount = $checkStageData->count();
												$localStageFlag = 1;
												$stageId = $stages['status_id'];
												$stageName = $stages['status_name'];
												$stageColorCode = $stages['color_code'];
												$stageDuration = $stages['duration'];
												$getStartNDuedate = $this->calculateDueDate($request, $response, $startingDate, $stageDuration);
												$stageData = [
													'job_id' => $productionJobLastId,
													'print_method_id' => ($stages['is_global'] == 0) ? $method['print_method_id'] : 0,
													'stages_id' => $stageId,
													'stage_name' => $stageName,
													'stage_color_code' => $stageColorCode,
													'created_date' => $startingDate,
													'starting_date' => ($stageKey == 0 && $checkStageDataCount == 0) ? $getStartNDuedate['start_date'] : '1000-01-01 00:00:00',
													'exp_completion_date' => ($stageKey == 0 && $checkStageDataCount == 0) ? $getStartNDuedate['due_date'] : '1000-01-01 00:00:00',
													'status' => ($stageKey == 0 && $checkStageDataCount == 0) ? 'in-progress' : 'not-started',
												];
												$productionJobStage = new ProductionJobStages($stageData);
												$productionJobStage->save();
												if ($stageKey == 0 && $checkStageDataCount == 0) {
													//update current stage in production table
													$currentStageId = $productionJobStage->xe_id;
													$productionJobInit = new ProductionJobs();
													$productionJobInit->where('xe_id', $productionJobLastId)
														->update([
															'current_stage_id' => $currentStageId,
														]);

													//Add assignee data
													$assignee = $this->saveAssigneeData($request, $productionJobLastId, $stageId, $currentStageId);
													if (!empty($assignee)) {
														$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
														$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
													}
													$tempSendingEmailData = [
														'customer_id' => '',
														'job_id' => $productionJobLastId,
														'stages_id' => '',
														'is_group' => $assignee['is_group'],
														'agent_ids' => $assignee['agent_id_arr'],
													];
													array_push($finalSendingEmailData, $tempSendingEmailData);
													//Adding to production job log for job creation
													$logData = [
														'job_id' => $productionJobLastId,
														'title' => 'Job assigned',
														'description' => 'Job #' . $jobId . ' assigned to ' . $type . ' ' . $names . ' for ' . $stageName . '.',
														'user_type' => $allPostPutVars['user_type'],
														'user_id' => $allPostPutVars['user_id'],
														'created_date' => date_time(
															'today', [], 'string'
														)
													];
													$this->addingProductionLog($logData);
												}
											}
										}
										//Save global stages
										//Get All Global Status
										if ($localStageFlag != 0) {
											$statusFeaturesInit = new StatusFeatures();
											$globalStatus = $statusFeaturesInit
												->select('production_status_features.status_id', 'production_status_features.duration', 'production_status.status_name', 'production_status.color_code')
												->join('production_status', 'production_status_features.status_id', '=', 'production_status.xe_id')
												->where(['production_status_features.is_global' => 1, 'production_status.store_id' => $getStoreDetails['store_id']])->orderBy('production_status.sort_order', 'ASC');
											if ($globalStatus->count() > 0) {
												$globalStatusArr = $globalStatus->get();
												foreach ($globalStatusArr as $globalStatus) {
													$globalStageData = [
														'job_id' => $productionJobLastId,
														'print_method_id' => 0,
														'stages_id' => $globalStatus['status_id'],
														'stage_name' => $globalStatus['status_name'],
														'stage_color_code' => $globalStatus['color_code'],
														'created_date' => $startingDate,
														'starting_date' => '1000-01-01 00:00:00',
														'exp_completion_date' => '1000-01-01 00:00:00',
														'status' => 'not-started',
													];
													$productionJobGlobalStage = new ProductionJobStages($globalStageData);
													$productionJobGlobalStage->save();
												}
											}
										} else {
											//Get All Global Status
											$statusFeaturesInit = new StatusFeatures();
											$globalStatus = $statusFeaturesInit
												->select('production_status_features.status_id', 'production_status_features.duration', 'production_status.status_name', 'production_status.color_code')
												->join('production_status', 'production_status_features.status_id', '=', 'production_status.xe_id')
												->where(['production_status_features.is_global' => 1,
													'production_status.store_id' => $getStoreDetails['store_id']])->orderBy('production_status.sort_order', 'ASC');
											if ($globalStatus->count() > 0) {
												$nonDecoGlobalStatusArr = $globalStatus->get();
												$nonDecoGlobalStatusArr = json_clean_decode($nonDecoGlobalStatusArr, true);
												$startingDate = date_time(
													'today', [], 'string'
												);
												$duration = $nonDecoGlobalStatusArr[0]['duration'];
												$expCompletionDate = date('Y-m-d H:i:s', strtotime('+' . $duration . ' hour', strtotime($startingDate)));
												$getStartNDuedate = $this->calculateDueDate($request, $response, $startingDate, $duration);
												//$finalSendingEmailData = [];
												foreach ($nonDecoGlobalStatusArr as $globalKey => $nonDecoGlobalStatus) {
													$globalStageData = [
														'job_id' => $productionJobLastId,
														'print_method_id' => 0,
														'stages_id' => $nonDecoGlobalStatus['status_id'],
														'stage_name' => $nonDecoGlobalStatus['status_name'],
														'stage_color_code' => $nonDecoGlobalStatus['color_code'],
														'created_date' => $startingDate,
														'starting_date' => ($globalKey == 0) ? $getStartNDuedate['start_date'] : '1000-01-01 00:00:00',
														'exp_completion_date' => ($globalKey == 0) ? $getStartNDuedate['due_date'] : '1000-01-01 00:00:00',
														'status' => ($globalKey == 0) ? 'in-progress' : 'not-started',
													];
													$productionJobGlobalStage = new ProductionJobStages($globalStageData);
													$productionJobGlobalStage->save();
													if ($globalKey == 0) {
														//update current stage in production table
														$currentStageId = $productionJobGlobalStage->xe_id;
														$productionJobInit = new ProductionJobs();
														$productionJobInit->where('xe_id', $productionJobLastId)
															->update([
																'current_stage_id' => $currentStageId,
															]);

														//Add assignee data
														$assignee = $this->saveAssigneeData($request, $productionJobLastId, $nonDecoGlobalStatus['status_id'], $currentStageId);
														if (!empty($assignee)) {
															$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
															$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
														}
														$tempSendingEmailData = [
															'customer_id' => '',
															'job_id' => $productionJobLastId,
															'stages_id' => '',
															'is_group' => $assignee['is_group'],
															'agent_ids' => $assignee['agent_id_arr'],
														];
														array_push($finalSendingEmailData, $tempSendingEmailData);

														//Adding to production job log for job creation
														$logData = [
															'job_id' => $productionJobLastId,
															'title' => 'Job assigned',
															'description' => 'Job #' . $jobId . ' assigned to ' . $type . ' ' . $names . ' for ' . $nonDecoGlobalStatus['status_name'] . '.',
															'user_type' => $allPostPutVars['user_type'],
															'user_id' => $allPostPutVars['user_id'],
															'created_date' => date_time(
																'today', [], 'string'
															)
														];
														$this->addingProductionLog($logData);
													}
												}
											}
										}
									} else {
										//Get All Global Status
										$statusFeaturesInit = new StatusFeatures();
										$globalStatus = $statusFeaturesInit
											->select('production_status_features.status_id', 'production_status_features.duration', 'production_status.status_name', 'production_status.color_code')
											->join('production_status', 'production_status_features.status_id', '=', 'production_status.xe_id')
											->where(['production_status_features.is_global' => 1, 'production_status.store_id' => $getStoreDetails['store_id']])->orderBy('production_status.sort_order', 'ASC');
										if ($globalStatus->count() > 0) {
											$nonDecoGlobalStatusArr = $globalStatus->get();
											$nonDecoGlobalStatusArr = json_clean_decode($nonDecoGlobalStatusArr, true);
											$startingDate = date_time(
												'today', [], 'string'
											);
											$duration = $nonDecoGlobalStatusArr[0]['duration'];
											$expCompletionDate = date('Y-m-d H:i:s', strtotime('+' . $duration . ' hour', strtotime($startingDate)));
											$getStartNDuedate = $this->calculateDueDate($request, $response, $startingDate, $duration);
											//$finalSendingEmailData = [];
											foreach ($nonDecoGlobalStatusArr as $globalKey => $nonDecoGlobalStatus) {
												$globalStageData = [
													'job_id' => $productionJobLastId,
													'print_method_id' => 0,
													'stages_id' => $nonDecoGlobalStatus['status_id'],
													'stage_name' => $nonDecoGlobalStatus['status_name'],
													'stage_color_code' => $nonDecoGlobalStatus['color_code'],
													'created_date' => $startingDate,
													'starting_date' => ($globalKey == 0) ? $getStartNDuedate['start_date'] : '1000-01-01 00:00:00',
													'exp_completion_date' => ($globalKey == 0) ? $getStartNDuedate['due_date'] : '1000-01-01 00:00:00',
													'status' => ($globalKey == 0) ? 'in-progress' : 'not-started',
												];
												$productionJobGlobalStage = new ProductionJobStages($globalStageData);
												$productionJobGlobalStage->save();
												if ($globalKey == 0) {
													//update current stage in production table
													$currentStageId = $productionJobGlobalStage->xe_id;
													$productionJobInit = new ProductionJobs();
													$productionJobInit->where('xe_id', $productionJobLastId)
														->update([
															'current_stage_id' => $currentStageId,
														]);

													//Add assignee data
													$assignee = $this->saveAssigneeData($request, $productionJobLastId, $nonDecoGlobalStatus['status_id'], $currentStageId);
													if (!empty($assignee)) {
														$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
														$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
													}
													$tempSendingEmailData = [
														'customer_id' => '',
														'job_id' => $productionJobLastId,
														'stages_id' => '',
														'is_group' => $assignee['is_group'],
														'agent_ids' => $assignee['agent_id_arr'],
													];
													array_push($finalSendingEmailData, $tempSendingEmailData);

													//Adding to production job log for job creation
													$logData = [
														'job_id' => $productionJobLastId,
														'title' => 'Job assigned',
														'description' => 'Job #' . $jobId . ' assigned to ' . $type . ' ' . $names . ' for ' . $nonDecoGlobalStatus['status_name'] . '.',
														'user_type' => $allPostPutVars['user_type'],
														'user_id' => $allPostPutVars['user_id'],
														'created_date' => date_time(
															'today', [], 'string'
														)
													];
													$this->addingProductionLog($logData);
												}
											}
										}
									}
								}
							}
							//Update production_status
							$orderInit->where([
								'order_id' => strval($orderId),
							])->update([
								'production_status' => 1,
								'customer_id' => $customerId,
								'product_count' => count($allProductArr),
							]);
						}
					}
				}

				$jsonResponse = [
					'status' => 1,
					'email_data' => $finalSendingEmailData,
					'message' => message('Production Job', 'saved'),
				];
			} else {
				$jsonResponse = [
					'status' => 0,
					'message' => message('Production Job', 'insufficient'),
				];
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Get Stages w.r.t Print Method
	 *
	 * @param $printMethodId  Print Method Id
	 *
	 * @author debashrib@riaxe.com
	 * @date   25 Sept 2020
	 * @return json response
	 */

	private function getStagesWrtPrintMethod($printMethodId) {
		$stagesDetails = [];
		if ($printMethodId > 0) {
			$statusPrintProfileRelInit = new StatusPrintProfileRel();
			$stagesData = $statusPrintProfileRelInit->select('status_id')->where('print_profile_id', $printMethodId);
			if ($stagesData->count() > 0) {
				$stagesDataArr = $stagesData->get();
				$stagesDataArr = json_clean_decode($stagesDataArr, true);
				foreach ($stagesDataArr as $stages) {
					$tempArr = $stages;
					$statusFeaturesInit = new StatusFeatures();
					$featureData = $statusFeaturesInit->where('status_id', $stages['status_id']);
					$productionStatusInit = new ProductionStatus();
					$statusData = $productionStatusInit->where('xe_id', $stages['status_id'])->orderBy('sort_order', 'ASC');
					$statusDataArr = $statusData->get();
					$tempArr['status_name'] = $statusDataArr[0]['status_name'];
					$tempArr['color_code'] = $statusDataArr[0]['color_code'];
					if ($featureData->count() > 0) {
						$featureDataArr = $featureData->get();
						$featureDataArr = json_clean_decode($featureDataArr, true);
						$tempArr['duration'] = $featureDataArr[0]['duration'];
						$tempArr['is_group'] = $featureDataArr[0]['is_group'];
						$tempArr['is_global'] = $featureDataArr[0]['is_global'];
					}
					array_push($stagesDetails, $tempArr);
				}
			}
		}
		return $stagesDetails;
	}

	/**
	 * Get All Print Method associated with order item
	 *
	 * @param $orderItemsArr  Order Items Array
	 *
	 * @author debashrib@riaxe.com
	 * @date   25 Sept 2020
	 * @return json response
	 */

	private function getPrintMethodOfOrder($orderItemsArr) {
		$printMethodArr = [];
		if (!empty($orderItemsArr)) {
			foreach ($orderItemsArr['decoration_settings_data'] as $settingData) {
				foreach ($settingData['decoration_data'] as $decorationData) {
					$tempPrintMethod['print_method_id'] = $decorationData['print_profile_id'];
					$tempPrintMethod['print_method_name'] = $decorationData['print_profile_name'];
					if (!in_array($tempPrintMethod, $printMethodArr)) {
						array_push($printMethodArr, $tempPrintMethod);
					}
				}
			}
		}
		return $printMethodArr;
	}

	/**
	 * Adding data to production log
	 *
	 * @param $logData  Log data array
	 *
	 * @author debashrib@riaxe.com
	 * @date   06 Apr 2020
	 * @return boolean
	 */
	public function addingProductionLog($logData) {
		if (!empty($logData)) {
			$productionLog = new ProductionJobLog($logData);
			if ($productionLog->save()) {
				return true;
			}
		}
		return false;
	}

	/**
	 * GET: Production Job List View
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   26 Sept 2020
	 * @return json response
	 */
	public function getProductionListView($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$productionJobInit = new ProductionJobs();
		// Collect all Filter columns from url
		$page = $request->getQueryParam('page');
		$perpage = $request->getQueryParam('perpage');
		$sortBy = $request->getQueryParam('sortby');
		$order = $request->getQueryParam('order');
		$productionJobStatus = $request->getQueryParam('production_job_status');
		$keyword = $request->getQueryParam('keyword');
		$stageId = $request->getQueryParam('stage_id');
		$from = $request->getQueryParam('from');
		$to = $request->getQueryParam('to');
		$agentIdArr = $request->getQueryParam('agent_id');
		$agentIdArr = json_clean_decode($agentIdArr, true);
		$orderIdArr = $request->getQueryParam('order_id');
		$orderIdArr = json_clean_decode($orderIdArr, true);
		$customerId = $request->getQueryParam('customer_id');
		$printMethodArr = $request->getQueryParam('print_methods');
		$printMethodArr = json_clean_decode($printMethodArr, true);
		$jobId = $request->getQueryParam('job_id');

		$productionJob = $productionJobInit
			->join('production_job_stages', 'production_jobs.current_stage_id', '=', 'production_job_stages.xe_id')
			->join('orders', 'production_jobs.order_id', '=', 'orders.order_id')
			->select('production_jobs.xe_id', 'production_jobs.store_id', 'production_jobs.job_id', 'production_jobs.order_id', 'production_jobs.order_item_id', 'production_jobs.order_item_quantity', 'production_jobs.job_title', 'production_jobs.job_status', 'production_jobs.note', 'production_jobs.comp_percentage', 'production_jobs.due_date', 'production_jobs.scheduled_date', 'production_jobs.created_at', 'production_jobs.current_stage_id', 'production_job_stages.xe_id as current_xe_id', 'production_job_stages.job_id as current_job_id', 'production_job_stages.print_method_id', 'production_job_stages.stages_id', 'production_job_stages.stage_name', 'production_job_stages.stage_color_code', 'production_job_stages.created_date', 'production_job_stages.starting_date', 'production_job_stages.exp_completion_date', 'production_job_stages.completion_date', 'production_job_stages.status', 'production_job_stages.message', 'orders.customer_id', 'orders.order_number')
			->where('production_jobs.store_id', $getStoreDetails['store_id']);

		//Filter by order
		if (isset($orderIdArr) && !empty($orderIdArr)) {
			$productionJob->whereIn('production_jobs.order_id', $orderIdArr);
		}
		//Filter by customer
		if (isset($customerId) && $customerId > 0) {
			$productionJob->where('orders.customer_id', $customerId);
		}

		//Filter by Production job
		if (isset($jobId) && $jobId > 0) {
			$productionJob->where('production_jobs.xe_id', $jobId);
		}

		//Filter by print method id
		if (isset($printMethodArr) && !empty($printMethodArr)) {
			$productionJob->whereIn('production_job_stages.print_method_id', $printMethodArr);
		}

		//Filter by keywords
		if (isset($keyword) && $keyword != "") {
			$productionJob->where('production_jobs.job_title', 'LIKE', '%' . $keyword . '%')
				->orWhere('production_jobs.job_id', 'LIKE', '%' . $keyword . '%')
				->orWhere('production_jobs.order_id', 'LIKE', '%' . $keyword . '%');
		}

		//Filter by expected completion date
		if (isset($from) && isset($to) && $from != "" && $to != "") {
			$to = date('Y-m-d H:i:s', strtotime($to . ' +1 day'));
			$productionJob->where('production_job_stages.exp_completion_date', '>=', $from)
				->where('production_job_stages.exp_completion_date', '<=', $to);
		}

		//Filter by current stage status
		if (isset($productionJobStatus) && $productionJobStatus != '') {
			$productionJob->where('production_job_stages.status', '=', $productionJobStatus);
		}

		//Filter by current stage id
		if (isset($stageId) && $stageId > 0) {
			$productionJob->where('production_job_stages.stages_id', '=', $stageId);
		}

		//Filter by agents
		if (isset($agentIdArr) && !empty($agentIdArr)) {
			$userRoleRelInit = new UserRoleRel();
			$userRole = $userRoleRelInit->select('role_id')->whereIn('user_id', $agentIdArr);
			$groupIdArr = [];
			$jobIdDataArr = [];
			if ($userRole->count() > 0) {
				$userRoleData = $userRole->get();
				$userRoleData = json_clean_decode($userRoleData, true);
				$groupIds = array_column($userRoleData, 'role_id');
				$groupIdArr = array_unique($groupIds);
			}
			$productionJobAgentInit = new ProductionJobAgents();
			$agentJobId = $productionJobAgentInit->select('job_id', 'job_stage_id')->whereIn('agent_id', $agentIdArr)
				->orWhere(function ($query) use ($groupIdArr) {
					$query->where('agent_id', $groupIdArr)
						->where('is_group', 1);
				});
			if ($agentJobId->count() > 0) {
				$jobIdData = $agentJobId->get();
				$jobIdData = json_clean_decode($jobIdData, true);
				foreach ($jobIdData as $id) {
					$finalJob = $productionJobInit->where([
						'xe_id' => $id['job_id'],
						'current_stage_id' => $id['job_stage_id'],
					]);
					if ($finalJob->count() > 0) {
						$finalData = $finalJob->get();
						$finalData = json_clean_decode($finalData, true);
						array_push($jobIdDataArr, $finalData[0]['xe_id']);
					}
				}
			}
			$productionJob->whereIn('production_jobs.xe_id', $jobIdDataArr);
		}

		$getTotalPerFilters = $productionJob->count();
		$offset = 0;
		if (isset($page) && $page != "") {
			$totalItem = empty($perpage) ? PAGINATION_MAX_ROW : $perpage;
			$offset = $totalItem * ($page - 1);
			$productionJob->skip($offset)->take($totalItem);
		}
		// Sorting by column name and sord order parameter
		if (isset($sortBy) && $sortBy != "" && isset($order) && $order != "") {
			$productionJob->orderBy($sortBy, $order);
		}
		if ($getTotalPerFilters > 0) {
			$getProductionJobData = [];
			$productionJobArr = $productionJob->get();
			foreach ($productionJobArr as $productionJob) {
				$tempProductionJob = $productionJob;
				//$currentStage = $this->getJobCurrentStage($productionJob['xe_id']);
				$tempProductionJob['order_number'] = ($tempProductionJob['order_number'] != '') ? $tempProductionJob['order_number'] : $tempProductionJob['order_id']; 
				$currentStage = [
					'xe_id' => $productionJob['current_xe_id'],
					'job_id' => $productionJob['current_job_id'],
					'print_method_id' => $productionJob['print_method_id'],
					'stages_id' => $productionJob['stages_id'],
					'status_name' => $productionJob['stage_name'],
					'status_color_code' => $productionJob['stage_color_code'],
					'created_date' => $productionJob['created_date'],
					'starting_date' => $productionJob['starting_date'],
					'exp_completion_date' => $productionJob['exp_completion_date'],
					'completion_date' => $productionJob['completion_date'],
					'status' => $productionJob['status'],
					'message' => $productionJob['message'],
				];
				unset($tempProductionJob['current_xe_id'], $tempProductionJob['current_job_id'], $tempProductionJob['print_method_id'], $tempProductionJob['stages_id'], $tempProductionJob['stage_name'], $tempProductionJob['stage_color_code'], $tempProductionJob['created_date'], $tempProductionJob['starting_date'], $tempProductionJob['exp_completion_date'], $tempProductionJob['completion_date'], $tempProductionJob['status'], $tempProductionJob['message']);
				//Get assignee
				$productionJobAgentInit = new ProductionJobAgents();
				$assigneeData = $productionJobAgentInit->select('is_group', 'agent_id')->where([
					'job_id' => $currentStage['job_id'],
					'job_stage_id' => $currentStage['xe_id'],
				]);
				$finalAssignee = [];
				if ($assigneeData->count() > 0) {
					$assigneeDataArr = $assigneeData->get();
					$assigneeDataArr = json_clean_decode($assigneeDataArr, true);
					foreach ($assigneeDataArr as $assignee) {
						array_push($finalAssignee, $assignee['agent_id']);
					}
				}
				$currentStage['is_group'] = $assigneeDataArr[0]['is_group'];
				$currentStage['assignee_data'] = $finalAssignee;
				$groupAgentId = [];
				if ($currentStage['is_group'] == 1 && is_array($allAgentArr) ) {
					foreach ($finalAssignee as $assignee) {
						$agentDetailsArr = array_filter($allAgentArr, function ($item) use ($assignee) {
							return $item['role_id'] == $assignee;
						});
						foreach ($agentDetailsArr as $agents) {
							array_push($groupAgentId, $agents['id']);
						}
					}
				}
				//Get PO Status
				$poStatus = '';
				$poStatusColorCode = '';
				$purchaseOrderItemInit = new PurchaseOrderItems();
				$purchaseOrderItem = $purchaseOrderItemInit
					->join('po_line_item_status', 'po_line_item_status.xe_id', '=', 'purchase_order_items.status_id')
					->select('po_line_item_status.status_name', 'po_line_item_status.color_code')
					->where([
						'purchase_order_items.order_id' => $productionJob['order_id'],
						'purchase_order_items.order_item_id' => $productionJob['order_item_id'],
					]);

				if ($purchaseOrderItem->count() > 0) {
					$poData = $purchaseOrderItem->get();
					$poData = json_clean_decode($poData, true);
					$poStatus = $poData[0]['status_name'];
					$poStatusColorCode = $poData[0]['color_code'];
				}
				$tempProductionJob['po_status'] = $poStatus;
				$tempProductionJob['po_status_color_code'] = $poStatusColorCode;
				$tempProductionJob['current_stage'] = $currentStage;
				$nextStageId = $this->getNextStageId($tempProductionJob['xe_id'], $tempProductionJob['current_stage_id']);
				$tempProductionJob['next_stage_id'] = ($nextStageId > 0) ? $nextStageId : '';
				$previousStageId = $this->getPreviousStageId($tempProductionJob['xe_id'], $tempProductionJob['current_stage_id']);
				$tempProductionJob['previous_stage_id'] = ($previousStageId > 0) ? $previousStageId : '';

				array_push($getProductionJobData, $tempProductionJob);
			}
			$jsonResponse = [
				'status' => 1,
				'records' => count($getProductionJobData),
				'total_records' => $getTotalPerFilters,
				'data' => $getProductionJobData,
			];
		} else {
			$jsonResponse = [
				'status' => 0,
				'message' => message('Production Job', 'not_found'),
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Get the current stage of production job
	 *
	 * @param $jobId  Production job id
	 *
	 * @author debashrib@riaxe.com
	 * @date   28 Sept 2020
	 * @return json response
	 */

	private function getJobCurrentStage($jobId) {
		$productionJobStageInit = new ProductionJobStages();
		$currentStage = [];
		if ($jobId > 0) {
			$jobStage = $productionJobStageInit
				->select('xe_id', 'job_id', 'print_method_id', 'stages_id', 'stage_name as status_name', 'stage_color_code as status_color_code', 'created_date', 'starting_date', 'exp_completion_date', 'completion_date', 'status', 'message')
				->where('job_id', $jobId)
				->where('status', '!=', 'not-started')
				->skip(0)->take(1)
				->orderBy('xe_id', 'DESC');
			if ($jobStage->count() > 0) {
				$jobStageArr = $jobStage->get();
				$jobStageArr = json_clean_decode($jobStageArr, true);
				$currentStage = $jobStageArr[0];
			}
		}
		return $currentStage;
	}

	/**
	 * GET: Production Job List View
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   26 Sept 2020
	 * @return json response
	 */
	public function getProductionCardView($request, $response) {
		$getStoreDetails = get_store_details($request);
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job', 'error'),
		];
		// Collect all Filter columns from url
		$page = $request->getQueryParam('page');
		$perpage = $request->getQueryParam('perpage');
		$orderId = $request->getQueryParam('order_id');
		$customerId = $request->getQueryParam('customer_id');
		$printMethodArr = $request->getQueryParam('print_methods');
		$printMethodArr = json_clean_decode($printMethodArr, true);
		$status = $request->getQueryParam('status');
		$agentId = $request->getQueryParam('agent_id');

		$orderInit = new Orders();
		$orderData = $orderInit
			->select('xe_id', 'order_id', 'artwork_status', 'order_status', 'po_status', 'production_status', 'production_percentage', 'store_id', 'customer_id', 'order_number','product_count', DB::raw('(SELECT created_at FROM production_jobs WHERE order_id = orders.order_id LIMIT 1) as created_at '), DB::raw('(SELECT count(xe_id) FROM production_jobs WHERE order_id = orders.order_id) as job_count '))
			->where('store_id', $getStoreDetails['store_id'])
			->where('production_status', '!=', '0');

		if (isset($orderId) && $orderId != '') {
			$orderData->where('order_id', $orderId);
		}

		if (isset($customerId) && $customerId != '') {
			$orderData->where('customer_id', $customerId);
		}

		//filter by status
		if (isset($status) && $status != '') {
			if ($status == 'completed') {
				$orderData->where('production_status', '2');
			} elseif ($status == 'inprogress') {
				$orderData->where('production_status', '1');
			}
		}

		if (isset($printMethodArr) && !empty($printMethodArr)) {
			$productionJobInit = new ProductionJobs();
			$productionJob = $productionJobInit
				->join('production_job_stages', 'production_jobs.current_stage_id', '=', 'production_job_stages.xe_id')
				->select('production_jobs.order_id')
				->where('production_jobs.store_id', $getStoreDetails['store_id'])
				->whereIn('production_job_stages.print_method_id', $printMethodArr);
			if ($productionJob->count() > 0) {
				$productionJobData = $productionJob->get();
				$productionJobData = json_clean_decode($productionJobData, true);
				$orderIdsArr = array_column($productionJobData, 'order_id');
				$finalOrderIdsArr = array_unique($orderIdsArr);
				if (!empty($finalOrderIdsArr)) {
					$orderData->whereIn('order_id', $finalOrderIdsArr);
				}
			}
		}

		if (isset($agentId) && !empty($agentId)) {
			$productionJobInit = new ProductionJobs();
			$userRoleRelInit = new UserRoleRel();
			$userRole = $userRoleRelInit->select('role_id')->where('user_id', $agentId);
			$groupIdArr = [];
			
			if ($userRole->count() > 0) {
				$userRoleData = $userRole->get();
				$userRoleData = json_clean_decode($userRoleData, true);
				$groupIds = array_column($userRoleData, 'role_id');
				$groupIdArr = array_unique($groupIds);
			}
			$productionJobAgentInit = new ProductionJobAgents();
			$agentJobId = $productionJobAgentInit->select('job_id', 'job_stage_id')->where('agent_id', $agentId)
				->orWhere(function ($query) use ($groupIdArr) {
					$query->where('agent_id', $groupIdArr)
						->where('is_group', 1);
				});
			if ($agentJobId->count() > 0) {
				$jobIdData = $agentJobId->get();
				$jobIdData = json_clean_decode($jobIdData, true);
				$jobIdArr = array_column($jobIdData, 'job_id');
				$job = $productionJobInit->select('order_id')->whereIn('xe_id', $jobIdArr);
				$orderIdData = $job->get()->toArray();
				$orderIdArr = array_column($orderIdData, 'order_id');
			}
			$orderData->whereIn('order_id', $orderIdArr);
		}

		$getTotalPerFilters = $orderData->count();
		$offset = 0;
		if (isset($page) && $page != "") {
			$totalItem = empty($perpage) ? PAGINATION_MAX_ROW : $perpage;
			$offset = $totalItem * ($page - 1);
			$orderData->skip($offset)->take($totalItem);
		}
		$orderData->orderBy('created_at', 'DESC');
		if ($getTotalPerFilters > 0) {
			$orderDataArr = $orderData->get();
			$cardViewData = [];
			foreach ($orderDataArr as $orders) {
				$orderId = $orders['order_id'];
				$tempOrderData['order_id'] = $orderId;
				$tempOrderData['order_number'] = ($orders['order_number'] != '') ? $orders['order_number'] : $orderId;
				$tempOrderData['order_production_status'] = ($orders['production_status'] == '1') ? 'In-Progress' : 'Completed';
				$tempOrderData['order_com_percentage'] = $orders['production_percentage'];
				$tempOrderData['job_count'] = $orders['job_count'];
				$tempOrderData['product_count'] = $orders['product_count'];
				$tempOrderData['items'] = [];
				array_push($cardViewData, $tempOrderData);
			}
			$jsonResponse = [
				'status' => 1,
				'records' => count($cardViewData),
				'total_records' => $getTotalPerFilters,
				'data' => $cardViewData,
			];
		} else {
			$jsonResponse = [
				'status' => 1,
				'data' => [],
				'message' => message('Production Job', 'not_found'),
			];
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Production Job List View
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   06 Oct 2020
	 * @return json response
	 */
	public function getCardViewDetails($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Production Job', 'error'),
        ];
        $getStoreDetails = get_store_details($request);
        $isProductCount = $request->getQueryParam('is_product_count');
        if (!empty($args['id'])) {
            $agentId = $request->getQueryParam('agent_id');
            $finalResult = [];
            $orderId = $args['id'];
            $ordersControllerInit = new OrdersController();
            $orderDetails = $ordersControllerInit->getOrderDetails($request, $response, $args, 1);
            $orderItems = $orderDetails['data']['orders'];
            //Get all Agent list 
            $userInit = new User();
            $getUser = $userInit->select('admin_users.xe_id as id', 'admin_users.name', 'admin_users.email', 'user_role_rel.role_id')
                ->join('user_role_rel', 'admin_users.xe_id', '=', 'user_role_rel.user_id') 
                ->where('admin_users.store_id', $getStoreDetails['store_id']);
            $allAgentArr = json_clean_decode($getUser->get(), true);
            $itemData = [];
            $finalGlobalStatus = [];
            $productionJobInit = new ProductionJobs();
            $productionJobStageInit = new ProductionJobStages();
            $globalProductionJob = $productionJobInit->where('order_id', $orderId);
            if ($globalProductionJob->count() > 0) {
                $globalProductionJob = $globalProductionJob->get();
                $globalProductionJob = json_clean_decode($globalProductionJob, true);
                $globalProductionJob = $globalProductionJob[0];
                $globalJobId = $globalProductionJob['xe_id'];
                $globalStageData = $productionJobStageInit->select('stages_id')->where([
                    'job_id' => $globalJobId,
                    'print_method_id' => 0
                ])->orderBy('xe_id', 'ASC');
                if ($globalStageData->count() > 0) {
                    $globalStageDataArr = $globalStageData->get();
                    $globalStageDataArr = json_clean_decode($globalStageDataArr, true);
                }
            }
            $allProductArr = [];
            foreach ($orderItems as $items) {
            	if ($isProductCount == 0 && !in_array($items['product_id'], $allProductArr)) {
	            	//check for product
					array_push($allProductArr, $items['product_id']);
				}
                $productionJob = $productionJobInit->where([
                    'order_id' => $orderId,
                    'order_item_id' => $items['id']
                ]);
                if ($productionJob->count() > 0) {
                    $productionJobData = $productionJob->get();
                    $productionJobData = json_clean_decode($productionJobData, true);
                    $tempItemData = $productionJobData[0];
                }
                $tempItemData['product_id'] = $items['product_id'];
                $tempItemData['product_name'] = $items['name'];
                $tempItemData['product_sku'] = $items['sku'];
                $tempItemData['quantity'] = $items['quantity'];
                $previousStageId = $this->getPreviousStageId($productionJobData[0]['xe_id'], $productionJobData[0]['current_stage_id']);
                $tempItemData['previous_stage_id'] = $previousStageId;
                //Get print methods
                $printMethodArr = $this->getPrintMethodOfOrder($items);
                $printMethodStagesData = [];
                foreach ($printMethodArr as $printMethod) {
                    $tempPrintMethodStagesData = $printMethod;
                    $productionJobStages = $productionJobStageInit->where([
                        'job_id' => $tempItemData['xe_id'],
                        'print_method_id' => $printMethod['print_method_id']
                    ])->orderBy('xe_id', 'ASC');
                    $finalStageData = [];
                    if ($productionJobStages->count() > 0) {
                        $productionJobStageArr = $productionJobStages->get();
                        $productionJobStageArr = json_clean_decode($productionJobStageArr, true);
                        foreach ($productionJobStageArr as $stageData) {
                            $tempStageData = $stageData;
                            //Get assignee
                            $productionJobAgentInit = new ProductionJobAgents();
                            $assigneeData = $productionJobAgentInit->where([
                                'job_id' => $stageData['job_id'],
                                'job_stage_id' => $stageData['xe_id']
                            ]);
                            $finalAssignee = [];
                            $finalIsGroup = [];
                            if ($assigneeData->count() > 0) {
                                $assigneeDataArr = $assigneeData->get();
                                foreach ($assigneeDataArr as $assignee) {
                                    array_push($finalAssignee, $assignee['agent_id']);
                                    array_push($finalIsGroup, $assignee['is_group']);
                                }
                            }
                            $tempStageData['is_group'] = $finalIsGroup[0];
                            $tempStageData['assignee_data'] = $finalAssignee;
                            $groupAgentId = [];
                            if ($finalIsGroup[0] == 1) {
                                foreach($finalAssignee as $assignee) {
                                    $agentDetailsArr = array_filter($allAgentArr, function ($item) use ($assignee) {
                                        return $item['role_id'] == $assignee;
                                    });
                                    foreach ($agentDetailsArr as $agents) {
                                        array_push($groupAgentId, $agents['id']);
                                    }
                                } 
                            }
                            //For agent view
                            if (isset($agentId) && $agentId != '' && $agentId > 0) {
                                if (in_array($agentId, $finalAssignee) || in_array($agentId, $groupAgentId)) {
                                    $tempStageData['is_agent_operate'] = 1;
                                } else {
                                    $tempStageData['is_agent_operate'] = 0;
                                }
                            }
                            $tempStageData['created_date'] = ($tempStageData['created_date'] != '0000-00-00 00:00:00' && $tempStageData['created_date'] != null) ? $tempStageData['created_date'] : '';
                            $tempStageData['starting_date'] = ($tempStageData['starting_date'] != '0000-00-00 00:00:00' && $tempStageData['starting_date'] != null) ? $tempStageData['starting_date'] : '';
                            $tempStageData['exp_completion_date'] = ($tempStageData['exp_completion_date'] != '0000-00-00 00:00:00' && $tempStageData['exp_completion_date'] != null) ? $tempStageData['exp_completion_date'] : '';
                            $tempStageData['completion_date'] = ($tempStageData['completion_date'] != '0000-00-00 00:00:00' && $tempStageData['completion_date'] != null) ? $tempStageData['completion_date'] : '';
                            array_push($finalStageData, $tempStageData);
                        }
                    }
                    if (!empty($finalStageData)) {
	                    $tempPrintMethodStagesData['stages_data'] = $finalStageData;
	                    array_push($printMethodStagesData, $tempPrintMethodStagesData);
	                }
                }
                $tempItemData['print_method'] = $printMethodStagesData;
                array_push($itemData, $tempItemData);
            }
            //Get Global Stages
            foreach ($globalStageDataArr as $globals) {
                $tempData['stage_id'] = $globals['stages_id'];
                $jobData = $productionJobInit->where([
                    'order_id' => $orderId
                ]);
                if ($jobData->count() > 0) {
                    $productionJobData = $jobData->get();
                    $productionJobData = json_clean_decode($productionJobData, true);
                    $globalStageDataArr = [];
                    foreach ($productionJobData as $productionJob) {
                        $orderItemId = $productionJob['order_item_id'];
                        $orderItemDetails = array_filter($orderItems, function ($item) use ($orderItemId) {
                            return $item['id'] == $orderItemId;
                        });
                        $orderItemDetails = $orderItemDetails[array_keys($orderItemDetails)[0]];
                        $globalStages = $productionJobStageInit
                        ->select('production_job_stages.xe_id', 'production_job_stages.job_id', 'production_job_stages.print_method_id', 'production_job_stages.stages_id', 'production_job_stages.stage_name', 'production_job_stages.stage_color_code', 'production_job_stages.created_date', 'production_job_stages.starting_date', 'production_job_stages.exp_completion_date', 'production_job_stages.completion_date', 'production_job_stages.status', 'production_job_stages.message', 'production_jobs.job_id as job_id_name', 'production_jobs.job_title', 'production_job_stages.completed_by')
                        ->join('production_jobs', 'production_job_stages.job_id', '=', 'production_jobs.xe_id')
                        ->where([
                            'production_job_stages.job_id' => $productionJob['xe_id'],
                            'production_job_stages.print_method_id' => 0,
                            'production_job_stages.stages_id' => $globals['stages_id']
                        ]);
                        if ($globalStages->count() > 0) {
                            $globalStageData = $globalStages->get();
                            $globalStageData = json_clean_decode($globalStageData, true);
                            $globalStageData = $globalStageData[0];
                            $firstStageName = $globalStageData['stage_name'];
                            $firstStageColorCode = $globalStageData['stage_color_code'];
                            $globalStageData['created_date'] = ($globalStageData['created_date'] != '0000-00-00 00:00:00' && $globalStageData['created_date'] != null) ? $globalStageData['created_date'] : '';
                            $globalStageData['starting_date'] = ($globalStageData['starting_date'] != '0000-00-00 00:00:00' && $globalStageData['starting_date'] != null) ? $globalStageData['starting_date'] : '';
                            $globalStageData['exp_completion_date'] = ($globalStageData['exp_completion_date'] != '0000-00-00 00:00:00' && $globalStageData['exp_completion_date'] != null) ? $globalStageData['exp_completion_date'] : '';
                            $globalStageData['completion_date'] = ($globalStageData['completion_date'] != '0000-00-00 00:00:00' && $globalStageData['completion_date'] != null) ? $globalStageData['completion_date'] : '';
                            $globalStageData['quantity'] = $orderItemDetails['quantity'];
                            //Get assignee
                            $productionJobAgentInit = new ProductionJobAgents();
                            $assigneeData = $productionJobAgentInit->where([
                                'job_id' => $productionJob['xe_id'],
                                'job_stage_id' => $productionJob['current_stage_id']
                            ]);
                            $finalAssignee = [];
                            $finalIsGroup = [];
                            if ($assigneeData->count() > 0) {
                                $assigneeDataArr = $assigneeData->get();
                                foreach ($assigneeDataArr as $assignee) {
                                    array_push($finalAssignee, $assignee['agent_id']);
                                    array_push($finalIsGroup, $assignee['is_group']);
                                }
                            }
                            $globalStageData['is_group'] = $finalIsGroup[0];
                            $globalStageData['assignee_data'] = $finalAssignee;
                            $groupAgentId = [];
                            if ($finalIsGroup[0] == 1) {
                                foreach($finalAssignee as $assignee) {
                                    $agentDetailsArr = array_filter($allAgentArr, function ($item) use ($assignee) {
                                        return $item['role_id'] == $assignee;
                                    });
                                    foreach ($agentDetailsArr as $agents) {
                                        array_push($groupAgentId, $agents['id']);
                                    }
                                } 
                            }
                            //For agent view
                            if (isset($agentId) && $agentId != '' && $agentId > 0) {
                                if (in_array($agentId, $finalAssignee) || in_array($agentId, $groupAgentId)) {
                                    $globalStageData['is_agent_operate'] = 1;
                                } else {
                                    $globalStageData['is_agent_operate'] = 0;
                                }
                            }
                            array_push($globalStageDataArr, $globalStageData);
                        }
                        
                    }
                }
                if (!empty($globalStageDataArr)) {
                    $tempData['stage_name'] = $firstStageName;
                    $tempData['stage_color_code'] = $firstStageColorCode;
                    //$tempData['assignee_data'] = [];
                    $tempData['stage_data'] = $globalStageDataArr;
                    array_push($finalGlobalStatus, $tempData);
                }
            }
            $finalResult['items'] = $itemData;
            $finalResult['global_status'] = $finalGlobalStatus;
            if ($isProductCount == 0) {
            	//Update product_count
            	$orderInit = new Orders();
				$orderInit->where([
					'order_id' => $orderId,
					'store_id' => $getStoreDetails['store_id'],
				])->update([
					'product_count' => count($allProductArr),
				]);
            	$finalResult['product_count'] = count($allProductArr);
            }
            
            $jsonResponse = [
                'status' => 1,
                'data' => $finalResult,
            ];
        }
        return response(
            $response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

	/**
	 * POST: Mark Production job stage as completed
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */
	public function productionJobStageOperations($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Stage Operation', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		if(!empty($args['current_stage_id']) && !empty($args['is_return']) ) {
			$productionJobStageId = $args['current_stage_id'];
			$nextProductionJobStageId = $args['next_stage_id'] ? $args['next_stage_id'] :0;
			$type = $args['type'];
			$isReturn = $args['is_return'];
		} else {
			$productionJobStageId = isset($allPostPutVars['current_stage_id']) ? $allPostPutVars['current_stage_id'] : 0;
			$type = isset($allPostPutVars['type']) ? $allPostPutVars['type'] : '';
			$nextProductionJobStageId = (isset($allPostPutVars['next_stage_id']) && $allPostPutVars['next_stage_id'] > 0) ? $allPostPutVars['next_stage_id'] : 0;
		}
		if ($productionJobStageId && $type != '') {
			$productionJobInit = new ProductionJobs();
			$productionJobStageInit = new ProductionJobStages();
			$orderInit = new Orders();
			$stageData = $productionJobStageInit->where([
				'xe_id' => $productionJobStageId,
			]);
			if ($stageData->count() > 0) {
				$stageDataArr = $stageData->get();
				$stageDataArr = json_clean_decode($stageDataArr, true);
				$stageDataArr = $stageDataArr[0];
				
				//Get Production Job Data
				$productionJobData = $this->getProductJobData($stageDataArr['job_id']);
				$orderId = $productionJobData['order_id'];
				$jobIdName = $productionJobData['job_id'];
				$customerId = $productionJobData['customer_id'];

				//Get Production Stage Data
				$productionStatusData = $this->getProductJobStageData($stageDataArr['stages_id']);
				$stageName = $productionStatusData['status_name'];
				//Get New stage details
				if ($nextProductionJobStageId > 0) {
					$nextStage = $productionJobStageInit->where('xe_id', $nextProductionJobStageId);
					$nextStageData = $nextStage->get();
					$nextStageData = json_clean_decode($nextStageData, true);
					$newStageName = $nextStageData[0]['stage_name'];
				}
				//Get All Print Profile
				$printMethodData = $this->getPrintMethodData($stageDataArr['print_method_id']);
				$printMethodName = $printMethodData['name'];

				if ($type == 'completed') {
					//Set Status to Completed
					$productionJobStageInit->where('xe_id', $productionJobStageId)
						->update([
							'completion_date' => date_time(
								'today', [], 'string'
							),
							'status' => 'completed',
							'completed_by' => $allPostPutVars['user_id'],
						]);

					//Start next job
					$assignee = [];
					if ($nextProductionJobStageId > 0) {
						$nextStageData = $productionJobStageInit->where('xe_id', $nextProductionJobStageId);
						if ($nextStageData->count() > 0) {
							$nextStageDetails = $nextStageData->get();
							$nextStageDetails = json_clean_decode($nextStageDetails, true);

							$nextProductionStatusData = $this->getProductJobStageData($nextStageDetails[0]['stages_id']);
							$nextDuration = $nextProductionStatusData['feature_data']['duration'];
							$nextStartingDate = date_time(
								'today', [], 'string'
							);
							$getStartNDuedate = $this->calculateDueDate($request, $response, $nextStartingDate, $nextDuration);
							//Start the next stage
							$productionJobStageInit->where('xe_id', $nextProductionJobStageId)
								->update([
									'starting_date' => $getStartNDuedate['start_date'],
									'exp_completion_date' =>  $getStartNDuedate['due_date'],
									'status' => 'in-progress',
								]);
							//Update the current stage in production
							$productionJobInit = new ProductionJobs();
							$productionJobInit->where('xe_id', $nextStageDetails[0]['job_id'])
								->update([
									'current_stage_id' => $nextProductionJobStageId,
								]);
							//Add assignee data
							$assignee = $this->saveAssigneeData($request, $nextStageDetails[0]['job_id'], $nextStageDetails[0]['stages_id'], $nextProductionJobStageId);
							//Adding to production job log
							if ($allPostPutVars['user_id'] == 1 && $allPostPutVars['user_type'] == 'admin') {
								$by = ' by Admin';
							} else {
								$userInit = new User();
								$agent = $userInit->select('xe_id', 'name')->where('xe_id', $allPostPutVars['user_id'])->first()->toArray();
								$by = ' by Agent '.$agent['name'];
							}
							if ($printMethodName != '') {
								if ($newStageName != '') {
									if (!empty($assignee)) {
										$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
										$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
									}
									$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . ' for print method ' . $printMethodName . $by.' and auto assigned to ' . $type . ' ' . $names . '.';
								} else {
									$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . ' for print method ' . $printMethodName . $by.'.';
								}
							} else {
								if ($newStageName != '') {
									if (!empty($assignee)) {
										$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
										$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
									}
									$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . $by.' and auto assigned to ' . $type . ' ' . $names . '.';
								} else {
									$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . $by.'.';
								}
							}
							$logData = [
								'job_id' => $stageDataArr['job_id'],
								'title' => 'Status changed and auto assigned',
								'description' => $description,
								'user_type' => !empty($allPostPutVars['user_type']) ? $allPostPutVars['user_type'] : 'admin',
								'user_id' => !empty($allPostPutVars['user_id']) ? $allPostPutVars['user_id'] : 1,
								'created_date' => date_time(
									'today', [], 'string'
								)
							];
							$this->addingProductionLog($logData);
						}
					}

					//Update the completed percentage
					$totalStages = $productionJobStageInit->where('job_id', $stageDataArr['job_id']);
					$totalStagesCount = $totalStages->count();
					$completedStages = $productionJobStageInit->where([
						'job_id' => $stageDataArr['job_id'],
						'status' => 'completed',
					]);
					$completedStagesCount = $completedStages->count();
					/*$percentage = ($completedStagesCount / $totalStagesCount) * 100;
					$productionJobInit->where('xe_id', $stageDataArr['job_id'])
						->update([
							'comp_percentage' => round($percentage),
						]);*/
					$getAllStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->where('production_job_stages.job_id', $stageDataArr['job_id']);
					$getAllStageData = $getAllStage->get()->toArray();
					$stageTotalTimeArr = array_column($getAllStageData, 'duration');
					$stageTotalTime = array_sum($stageTotalTimeArr);
					$completedStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->where('production_job_stages.status', 'completed')->where('production_job_stages.job_id', $stageDataArr['job_id']);

					$completedStageData = $completedStage->get()->toArray();
					$stageCompTimeArr = array_column($completedStageData, 'duration');
					$stageCompTime = array_sum($stageCompTimeArr);
					$percentage = ($stageCompTime / $stageTotalTime) * 100;
					$productionJobInit->where('xe_id', $stageDataArr['job_id'])
					->update([
						'comp_percentage' => round($percentage),
					]);
					//Mark the job as completed
					if ($totalStagesCount == $completedStagesCount) {
						$productionJobInit->where('xe_id', $stageDataArr['job_id'])
							->update([
								'job_status' => 'completed',
							]);
						//Add to production log
						$logData = [
							'job_id' => $stageDataArr['job_id'],
							'title' => '<span  class="text-success">Job completed</span>',
							'description' => '<span  class="text-success">Job #' . $jobIdName . ' is completed.</span>',
							'user_type' => !empty($allPostPutVars['user_type']) ? $allPostPutVars['user_type'] : 'admin',
							'user_id' => !empty($allPostPutVars['user_id']) ? $allPostPutVars['user_id'] : 1,
							'created_date' => date_time(
								'today', [], 'string'
							)
						];
						$this->addingProductionLog($logData);
						//Change the order completed percentage
						$allProductionJob = $productionJobInit->where('order_id', $orderId);
						$totalProductionJobCount = $allProductionJob->count();
						$complatedProductionJob = $productionJobInit->where([
							'order_id' => $orderId,
							'job_status' => 'completed',
						]);
						$complatedProductionJobCount = $complatedProductionJob->count();
						/*$orderPercentage = ($complatedProductionJobCount / $totalProductionJobCount) * 100;
						$orderInit->where('order_id', $orderId)
							->update([
								'production_percentage' => round($orderPercentage),
							]);*/

						if ($totalProductionJobCount == $complatedProductionJobCount) {
							$orderInit->where('order_id', $orderId)
								->update([
									'production_status' => 2,
								]);
						}
					}
					$getAllJob = $productionJobInit->select('xe_id')->where('order_id', $orderId);
					$getAllJobData = $getAllJob->get()->toArray();
					$jobIdArr = array_column($getAllJobData, 'xe_id');
					$getAllStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->whereIn('production_job_stages.job_id', $jobIdArr);
					$getAllStageData = $getAllStage->get()->toArray();
					$stageTotalTimeArr = array_column($getAllStageData, 'duration');
					$stageTotalTime = array_sum($stageTotalTimeArr);
					$completedStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->where('production_job_stages.status', 'completed')->whereIn('production_job_stages.job_id', $jobIdArr);

					$completedStageData = $completedStage->get()->toArray();
					$stageCompTimeArr = array_column($completedStageData, 'duration');
					$stageCompTime = array_sum($stageCompTimeArr);
					$orderPercentage = ($stageCompTime / $stageTotalTime) * 100;
					$orderInit->where('order_id', $orderId)
					->update([
						'production_percentage' => round($orderPercentage),
					]);
				}
			}
			$currentOrder = $orderInit->where('order_id', $orderId);
			$currentOrderData = $currentOrder->get();
			$currentOrderData = json_clean_decode($currentOrderData, true);
			$orderStatus = ($currentOrderData[0]['production_status'] == '1') ? 'In-Progress' : 'Completed';
			$orderPercentage = $currentOrderData[0]['production_percentage'];
			$sendingEmailData = [
				'customer_id' => $customerId,
				'job_id' => $stageDataArr['job_id'],
				'stages_id' => $stageDataArr['stages_id'],
				'is_group' => ($nextProductionJobStageId > 0) ? $assignee['is_group'] : '',
				'agent_ids' => ($nextProductionJobStageId > 0) ? $assignee['agent_id_arr'] : [],
			];
			$jsonResponse = [
				'status' => 1,
				'order_production_status' => $orderStatus,
				'order_com_percentage' => $orderPercentage,
				'email_data' => $sendingEmailData,
				'message' => message('Stage Operation', 'done'),
			];
			if($isReturn== 1) {
				return $jsonResponse;
			}
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}
	/**
	 * Get production job data
	 *
	 * @param $jobId  Production job id
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */

	private function getProductJobData($jobId) {
		$productionJobDataArr = [];
		if ($jobId != '' && $jobId > 0) {
			$productionJobInit = new ProductionJobs();
			$productionJob = $productionJobInit
				->select('production_jobs.xe_id', 'production_jobs.job_id', 'production_jobs.order_id', 'production_jobs.order_item_id', 'production_jobs.order_item_quantity', 'production_jobs.job_title', 'production_jobs.job_status', 'production_jobs.note', 'production_jobs.comp_percentage', 'production_jobs.due_date', 'production_jobs.scheduled_date', 'production_jobs.created_at', 'production_jobs.current_stage_id', 'orders.customer_id')
				->join('orders', 'production_jobs.order_id', '=', 'orders.order_id')
				->where('production_jobs.xe_id', $jobId);
			$productionJobData = $productionJob->get();
			$productionJobData = json_clean_decode($productionJobData, true);
			$productionJobDataArr = $productionJobData[0];
		}
		return $productionJobDataArr;
	}

	/**
	 * Get production job stage data
	 *
	 * @param $stageId  Production job stage id
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */

	private function getProductJobStageData($stageId) {
		$productionJobStageDataArr = [];
		if ($stageId != '' && $stageId > 0) {
			$productionStatusInit = new ProductionStatus();
			$productionStatus = $productionStatusInit->where('xe_id', $stageId);
			$productionStatusData = $productionStatus->get();
			$productionStatusData = json_clean_decode($productionStatusData, true);
			$productionJobStageDataArr = $productionStatusData[0];
			// Get Feature Data
			$stageFeatureDataArr = [];
			$statusFeatureInit = new StatusFeatures();
			$stageFeature = $statusFeatureInit->where('status_id', $stageId);
			if ($stageFeature->count() > 0) {
				$stageFeatureData = $stageFeature->get();
				$stageFeatureData = json_clean_decode($stageFeatureData, true);
				$stageFeatureDataArr = $stageFeatureData[0];
			}
			$productionJobStageDataArr['feature_data'] = $stageFeatureDataArr;
		}
		return $productionJobStageDataArr;
	}

	/**
	 * Get print method data
	 *
	 * @param $printMethodId  Print method id
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */

	private function getPrintMethodData($printMethodId) {
		$printMethodDataArr = [];
		if ($printMethodId != '' && $printMethodId > 0) {
			$printProfileInit = new PrintProfile();
			$printMethodData = $printProfileInit->select('name')->where('xe_id', $printMethodId)->first();
			$printMethodDataArr = json_clean_decode($printMethodData, true);
		}
		return $printMethodDataArr;
	}

	/**
	 * POST: Production job stage dealy
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */
	public function productionJobStageDelayed($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Stage Delayed', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$productionJobStageInit = new ProductionJobStages();
		$today = date_time(
			'today', [], 'string'
		);
		$productionJob = $productionJobStageInit->where('status', 'in-progress')
			->where('exp_completion_date', '<', $today);
		$productionJobIdArr = [];
		if ($productionJob->count() > 0) {
			$productionJobArr = $productionJob->get();
			foreach ($productionJobArr as $jobData) {
				$jobId = $jobData['job_id'];
				$printMethodId = $jobData['print_method_id'];
				$stageId = $jobData['stages_id'];
				//Update status as delayed
				$productionJobStageInit->where('xe_id', $jobData['xe_id'])
					->update([
						'status' => 'delay',
					]);
				//Get Production Job Data
				$productionJobData = $this->getProductJobData($jobId);
				$jobIdName = $productionJobData['job_id'];
				//Get Production Stage Data
				$productionStatusData = $this->getProductJobStageData($stageId);
				//Get All Print Profile
				$printMethodData = $this->getPrintMethodData($printMethodId);
				$printMethodName = $printMethodData['name'];
				if ($printMethodName != '') {
					$description = '<span  class="text-danger">Due date of job #' . $jobIdName . ' for print method ' . $printMethodName . ' is over and status set as "Delayed".</span>';
				} else {
					$description = '<span  class="text-danger">Due date of job #' . $jobIdName . ' is over and status set as "Delayed".</span>';
				}
				$logData = [
					'job_id' => $jobId,
					'title' => '<span  class="text-danger">Overdue Alert</span>',
					'description' => $description,
					'user_type' => $allPostPutVars['user_type'],
					'user_id' => $allPostPutVars['user_id'],
					'created_date' => date_time(
						'today', [], 'string'
					)
				];
				$this->addingProductionLog($logData);
				$tempData['production_job_stage_id'] = $jobData['xe_id'];
				$tempData['status'] = 'delay';
				//Get stage is global or not
				$statusFeatureInit = new StatusFeatures();
				$statusFeature = $statusFeatureInit->where('status_id', $stageId);
				if ($statusFeature->count() > 0) {
					$statusFeatureData = $statusFeature->get();
					$statusFeatureData = json_clean_decode($statusFeatureData, true);
					$isGlobal = $statusFeatureData[0]['is_global'];
					$tempData['is_global'] = $isGlobal;
				}

				array_push($productionJobIdArr, $tempData);
			}
		}
		$jsonResponse = [
			'status' => 1,
			'data' => $productionJobIdArr,
			'message' => message('Stage Delayed', 'done'),
		];
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Production Job Log
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */
	public function getProductionJobLogs($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Log', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		if (!empty($args['id'])) {
			$productionJobId = $args['id'];
			$productionJobLogInit = new ProductionJobLog();
			$logData = $productionJobLogInit->where('job_id', $productionJobId)
				->orderBy('created_date', 'DESC');
			$noteRes = [];
			$finalLogData = [];
			if ($logData->count() > 0) {
				$logDataArr = $logData->get();
				foreach ($logDataArr as $logs) {
					$tempLogData = $logs;
					$userName = $logs['user_type'];
					if ($logs['user_type'] == 'agent') {
						//Get agent name
						$userInit = new User();
						$agent = $userInit->select('xe_id', 'name')->where('xe_id', $logs['user_id']);
						if ($agent->count() > 0) {
                            $agentDetails = json_clean_decode($agent->first(), true);
                            $userName = $agentDetails['name'];
                        }
					} elseif ($logs['user_type'] == 'customer') {
						//Get customer details
						$customersControllerInit = new CustomersController();
						$customerDetails = $customersControllerInit->getQuoteCustomerDetails($logs['user_id'], $getStoreDetails['store_id'], '');
                        if (!empty($customerDetails)) {
                            $userName = ($customerDetails['customer']['name'] != '') ? $customerDetails['customer']['name'] : $customerDetails['customer']['email'];
                        }
					}
					$tempLogData['user_name'] = $userName;
					$tempLogData['title'] = stripslashes($logs['title']);
					$tempLogData['description'] = stripslashes($logs['description']);
					$tempLogData['created_at'] = $logs['created_date'];
					$tempLogData['log_type'] = 'job_log';
					unset(
						$logs['created_date']
					);
					array_push($finalLogData, $tempLogData);
				}

			}
			//Get internal note data
			$internalNoteInit = new ProductionJobNotes();
			$internalNotes = $internalNoteInit->with('files')->where('job_id', $productionJobId)
				->orderBy('created_date', 'DESC');
			if ($internalNotes->count() > 0) {
				$noteDataArr = $internalNotes->get();
				foreach ($noteDataArr as $noteData) {
					$newNoteArr = $noteData;
					$userName = $newNoteArr['user_type'];
					if ($newNoteArr['user_type'] == 'agent') {
						//Get agent name
						$userInit = new User();
						$agent = $userInit->select('xe_id', 'name')->where('xe_id', $newNoteArr['user_id']);
						if ($agent->count() > 0) {
                            $agentDetails = json_clean_decode($agent->first(), true);
                            $userName = $agentDetails['name'];
                        }
					}
					$newNoteArr['title'] = 'Internal note added by ' . $userName;
					$newNoteArr['description'] = $newNoteArr['note'];
					$newNoteArr['log_type'] = 'internal_note';
					$newNoteArr['user_name'] = $userName;
					$newNoteArr['created_at'] = $newNoteArr['created_date'];
					unset(
						$newNoteArr['note'],
						$newNoteArr['seen_flag'],
						$newNoteArr['created_date']
					);
					array_push($noteRes, $newNoteArr);
				}
			}
			$totalProductionJobLogs = array_merge($finalLogData, $noteRes);
			// Sort the array by Created Date and time
			usort($totalProductionJobLogs, 'date_compare');
			if (is_array($totalProductionJobLogs) && !empty($totalProductionJobLogs) > 0) {
				$jsonResponse = [
					'status' => 1,
					'data' => $totalProductionJobLogs,
				];
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Production Job Details
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   29 Sept 2020
	 * @return json response
	 */
	public function getProductionJobDetails($request, $response, $args, $returnType = 0) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Log', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$currentStageId = $request->getQueryParam('current_stage_id');
		if (!empty($args['id'])) {
			$productionJobId = $args['id'];
			$productionJobInit = new ProductionJobs();

			$productionJob = $productionJobInit->where([
				'store_id' => $getStoreDetails['store_id'],
				'xe_id' => $productionJobId,
			]);

			$finalProductJobData = [];
			$productionJobArr = $productionJob->first();
			$productionJobArr = json_clean_decode($productionJobArr, true);
			if (!empty($productionJobArr)) {
				$orderId = $productionJobArr['order_id'];
				$orderItemId = $productionJobArr['order_item_id'];
				//Get Order data
				$orderData = $this->getOrderItemData($getStoreDetails['store_id'], $orderId, $orderItemId);
				$finalProductJobData['product_image'] = $orderData['orders']['store_image'];
				$finalProductJobData['order_data'] = $orderData;
				$currentStage = $this->getJobCurrentStage($productionJobArr['xe_id']);
				$userInit = new User();
				$allAgent = $userInit->select('xe_id as id', 'name')->where('store_id', $getStoreDetails['store_id']);
				$allAgentArr = json_clean_decode($allAgent->get(), true);
				$userRoleInit = new UserRole();
				$allGroup = $userRoleInit->select('xe_id as id', 'role_name')->where('store_id', $getStoreDetails['store_id']);
				$allGroupArr = json_clean_decode($allGroup->get(), true);

				$productionJobAgentInit = new ProductionJobAgents();
				$assigneeData = $productionJobAgentInit->where([
					'job_id' => $currentStage['job_id'],
					'job_stage_id' => $currentStage['xe_id'],
				]);
				$finalAssignee = [];
				$finalIsGroup = [];
				$assigneeDataArr = $assigneeData->get();
				if (!empty($assigneeDataArr)) {
					foreach ($assigneeDataArr as $assignee) {
						if ($assignee['is_group'] == 0) {
							$agentId = $assignee['agent_id'];
							$agentDetails = array_filter($allAgentArr, function ($item) use ($agentId) {
								return $item['id'] == $agentId;
							});
							$agentDetails = $agentDetails[array_keys($agentDetails)[0]];
							$tempData['id'] = $agentId;
							$tempData['name'] = $agentDetails['name'];
						} else {
							$groupId = $assignee['agent_id'];
							$groupDetails = array_filter($allGroupArr, function ($item) use ($groupId) {
								return $item['id'] == $groupId;
							});
							$groupDetails = $groupDetails[array_keys($groupDetails)[0]];
							$tempData['id'] = $groupId;
							$tempData['name'] = $groupDetails['role_name'];
						}
						array_push($finalAssignee, $tempData);
						array_push($finalIsGroup, $assignee['is_group']);
					}
				}
				$currentStage['is_group'] = $finalIsGroup[0];
				$currentStage['assignee_data'] = $finalAssignee;
				$nextStageId = $this->getNextStageId($productionJobArr['xe_id'], $productionJobArr['current_stage_id']);
				$productionJobArr['next_stage_id'] = ($nextStageId > 0) ? $nextStageId : '';
				$previousStageId = $this->getPreviousStageId($productionJobArr['xe_id'], $productionJobArr['current_stage_id']);
				$productionJobArr['previous_stage_id'] = ($previousStageId > 0) ? $previousStageId : '';
				$poStatus = '';
				$poStatusColorCode = '';
				$purchaseOrderItemInit = new PurchaseOrderItems();
				$purchaseOrderItem = $purchaseOrderItemInit
					->join('po_line_item_status', 'po_line_item_status.xe_id', '=', 'purchase_order_items.status_id')
					->select('po_line_item_status.status_name', 'po_line_item_status.color_code')
					->where([
						'purchase_order_items.order_id' => $productionJobArr['order_id'],
						'purchase_order_items.order_item_id' => $productionJobArr['order_item_id'],
					]);
				$poData = $purchaseOrderItem->first();
				$poData = json_clean_decode($poData, true);
				if (!empty($poData)) {
					$poStatus = $poData['status_name'];
					$poStatusColorCode = $poData['color_code'];
				}
				$productionJobArr['po_status'] = $poStatus;
				$productionJobArr['po_status_color_code'] = $poStatusColorCode;
				$productionJobArr['current_stage'] = $currentStage;
				if ($currentStageId != '') {
					$productionJobStagesInit = new ProductionJobStages();
					$getStageData = $productionJobStagesInit->select('status', 'stage_name')->where([
						'job_id' => $productionJobId,
						'xe_id' => $currentStageId
					]);
					$showMarkAsDone = true;
					$stageDataArr = $getStageData->first();
					$stageDataArr = json_clean_decode($stageDataArr, true);
					if (!empty($stageDataArr) && in_array($stageDataArr['status'],['completed','not-started'])) {
						$showMarkAsDone = false;
					}
					$productionJobArr['show_mark_as_done'] = $showMarkAsDone;
					$productionJobArr['qr_current_stage'] = $stageDataArr['stage_name'];
				}
				$token = 'job_id=' . $productionJobId.'&current_stage_id='.$currentStage['xe_id'].'&store_id='.$getStoreDetails['store_id'];
	            $token = base64_encode($token);
	            $url = 'quotation/production-job?token=' . $token;
	            $url = API_URL . $url;
				$productionJobArr['qr_code_url'] = $url;
				$finalProductJobData['production_job'] = $productionJobArr;
			}
			$jsonResponse = [
				'status' => 1,
				'data' => $finalProductJobData,
			];
			if ($returnType == 1) {
				return $finalProductJobData;
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Get Order Data
	 *
	 * @param $orderId  Order Id
	 * @param $itemId  Order Item Id
	 *
	 * @author debashrib@riaxe.com
	 * @date   30 Sept 2020
	 * @return array response
	 */

	private function getOrderItemData($storeId, $orderId, $itemId) {
		if ($orderId != '' && $orderId > 0 && $itemId != '' && $itemId > 0) {
			$orderInit = new OrdersController();
			$storeResponse = $orderInit->getStoreOrderLineItemDetails($orderId, $itemId, $is_customer = true, $storeId);
			$getdesignData['order_id'] = $storeResponse['order_id'];
			$getdesignData['order_number'] = $storeResponse['order_number'];
			$getdesignData['id'] = $storeResponse['item_id'];
			$getdesignData['product_id'] = $storeResponse['product_id'];
			$getdesignData['variant_id'] = $storeResponse['variant_id'];
			$getdesignData['name'] = $storeResponse['name'];
			$getdesignData['quantity'] = $storeResponse['quantity'];
			$getdesignData['sku'] = $storeResponse['sku'];
			$productsControllerInit = new ProductsController();
			$productIdForDesc = (strtolower(STORE_NAME) == "shopify") ? $storeResponse['variant_id'] : $storeResponse['product_id'];
			$getdesignData['description'] = $productsControllerInit->getStore()->getProductDescription($productIdForDesc);//$storeResponse['description'];
			$getdesignData['price'] = $storeResponse['price'];
			$getdesignData['total'] = $storeResponse['total'];
			$getdesignData['images'] = $storeResponse['images'];
			$getdesignData['custom_design_id'] = $storeResponse['custom_design_id'];
			$getdesignData['item_id'] = $itemId;
			$designData[] = $getdesignData;
			$designDataResponse = $orderInit->getOrderItemDesignData($designData, $storeId);
			$itemDataArr['id'] = $storeResponse['order_id'];
			$itemDataArr['order_number'] = $storeResponse['order_number'];
			$itemDataArr['total_amount'] = $storeResponse['total'];
			$itemDataArr['customer_first_name'] = ($storeResponse['customer_first_name']) ? $storeResponse['customer_first_name'] : $storeResponse['billing']['first_name'];
			$itemDataArr['customer_last_name'] = ($storeResponse['customer_last_name']) ? $storeResponse['customer_last_name'] : $storeResponse['billing']['last_name'];
			$itemDataArr['customer_email'] = $storeResponse['customer_email'];
			$itemDataArr['customer_id'] = $storeResponse['customer_id'];
			$itemDataArr['billing'] = $storeResponse['billing'];
			$itemDataArr['shipping'] = $storeResponse['shipping'];
			$itemDataArr['orders'] = $designDataResponse[0];
		}
		return $itemDataArr;
	}

	/**
	 * Add assignee data
	 *
	 * @param $jobId  Production job id
	 * @param $stageId  Production job status id
	 * @param $jobStageId  Production job stage id
	 *
	 * @author debashrib@riaxe.com
	 * @date   01 Oct 2020
	 * @return array response
	 */

	private function saveAssigneeData($request, $jobId, $stageId, $jobStageId) {
		$result = [];
		$getStoreDetails = get_store_details($request);
		if ($jobId != '' && $jobId > 0 && $stageId != '' && $stageId > 0 && $jobStageId != '' && $jobStageId > 0) {
			//Get assignee
			$statusAssigneeRelInit = new StatusAssigneeRel();
			$assigneeData = $statusAssigneeRelInit->where('status_id', $stageId);
			$finalAssignee = [];
			$success = 0;
			if ($assigneeData->count() > 0) {
				$assigneeDataArr = $assigneeData->get();
				$userInit = new User();
				$allAgent = $userInit->select('xe_id as id', 'name')->where('store_id', $getStoreDetails['store_id']);
				$agentListArr = json_clean_decode($allAgent->get(), true);
				//Get all group name
				$userRoleInit = new UserRole();
				$allGroup = $userRoleInit->select('xe_id as id', 'role_name')->where('store_id', $getStoreDetails['store_id']);
				$allGroupArr = json_clean_decode($allGroup->get(), true);
				foreach ($assigneeDataArr as $assignee) {
					$tempData['agent_id'] = $assignee['assignee_id'];
					//Get stage is global or not
					$statusFeatureInit = new StatusFeatures();
					$statusFeature = $statusFeatureInit->where('status_id', $stageId);
					if ($statusFeature->count() > 0) {
						$statusFeatureData = $statusFeature->get();
						$statusFeatureData = json_clean_decode($statusFeatureData, true);
						$isGroup = $statusFeatureData[0]['is_group'];
						$tempData['is_group'] = $isGroup;
					}
					array_push($finalAssignee, $tempData);
				}
				//Save assignee data in 'production_job_agents' table
				$allAgentsName = '';
				$agentIdsArr = [];
				foreach ($finalAssignee as $assigneeData) {
					$saveData = [
						'job_id' => $jobId,
						'job_stage_id' => $jobStageId,
						'is_group' => $assigneeData['is_group'],
						'agent_id' => $assigneeData['agent_id'],
					];
					$productionJobAgent = new ProductionJobAgents($saveData);
					if ($productionJobAgent->save()) {
						$success++;
						$isGroup = $assigneeData['is_group'];
						$agentId = $assigneeData['agent_id'];
						if ($isGroup == 0) {
							$agentArr = array_filter($agentListArr, function ($item) use ($agentId) {
								return $item['id'] == $agentId;
							});
							$agentArr = $agentArr[array_keys($agentArr)[0]];
							$allAgentsName .= ', ' . $agentArr['name'];
						} else {
							$groupDetails = array_filter($allGroupArr, function ($item) use ($agentId) {
								return $item['id'] == $agentId;
							});
							$groupDetails = $groupDetails[array_keys($groupDetails)[0]];
							$allAgentsName .= ', ' . $groupDetails['role_name'];
						}
						array_push($agentIdsArr, $agentId);
					}
				}
			}
		}
		if ($success > 0) {
			$result = [
				'is_group' => $isGroup,
				'agent_id_arr' => $agentIdsArr,
				'names' => trim($allAgentsName, ", "),
			];
		}
		return $result;

	}

	/**
	 * POST: Change Assignee
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   01 Oct 2020
	 * @return json response
	 */
	public function changeStageAssignee($request, $response) {
		$getStoreDetails = get_store_details($request);
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Stage Assignee', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$jobId = (isset($allPostPutVars['job_id']) && $allPostPutVars['job_id'] != '') ? $allPostPutVars['job_id'] : 0;
		$jobStageId = (isset($allPostPutVars['stage_id']) && $allPostPutVars['stage_id'] != '') ? $allPostPutVars['stage_id'] : 0;
		$agentIdArr = (isset($allPostPutVars['agent_ids']) && !empty($allPostPutVars['agent_ids'])) ? $allPostPutVars['agent_ids'] : [];
		$agentIdArr = json_clean_decode($agentIdArr, true);
		if ($jobId > 0 && $jobStageId > 0 && !empty($agentIdArr)) {
			//Get production job stage data
			$productionJobStageInit = new ProductionJobStages();
			$productionJobStageData = $productionJobStageInit->where('xe_id', $jobStageId);
			if ($productionJobStageData->count() > 0) {
				$productionJobStageDataArr = $productionJobStageData->get();
				$productionJobStageDataArr = json_clean_decode($productionJobStageDataArr, true);
			}

			$productionJobAgentInit = new ProductionJobAgents();
			$productionJobAgent = $productionJobAgentInit->where([
				'job_id' => $jobId,
				'job_stage_id' => $jobStageId,
			]);
			if ($productionJobAgent->count() > 0) {
				$productionJobAgentData = $productionJobAgent->get();
				$productionJobAgentData = json_clean_decode($productionJobAgentData, true);
				$userInit = new User();
				$allAgent = $userInit->select('xe_id as id', 'name')->where('store_id', $getStoreDetails['store_id']);
				$agentListArr = json_clean_decode($allAgent->get(), true);
				//Get all group name
				$userRoleInit = new UserRole();
				$allGroup = $userRoleInit->select('xe_id as id', 'role_name')->where('store_id', $getStoreDetails['store_id']);
				$allGroupArr = json_clean_decode($allGroup->get(), true);

				//Delete data from table
				$deleteAgents = $productionJobAgentInit->where([
					'job_id' => $jobId,
					'job_stage_id' => $jobStageId,
				]);
				$deleteAgents->delete();
				//Save new agents

				$allAgentsName = '';
				$agentIdsArr = [];
				foreach ($agentIdArr as $agentIds) {
					$saveData = [
						'job_id' => $jobId,
						'job_stage_id' => $jobStageId,
						'is_group' => $allPostPutVars['is_group'],
						'agent_id' => $agentIds,
					];
					$productionJobAgent = new ProductionJobAgents($saveData);
					$productionJobAgent->save();

					//Send Email to agent
					if ($allPostPutVars['is_group'] == 0) {
						$agentArr = array_filter($agentListArr, function ($item) use ($agentIds) {
							return $item['id'] == $agentIds;
						});
						$agentArr = $agentArr[array_keys($agentArr)[0]];
						$allAgentsName .= ', ' . $agentArr['name'];
					} else {
						$groupDetails = array_filter($allGroupArr, function ($item) use ($agentIds) {
							return $item['id'] == $agentIds;
						});
						$groupDetails = $groupDetails[array_keys($groupDetails)[0]];
						$allAgentsName .= ', ' . $groupDetails['role_name'];
					}
					array_push($agentIdsArr, $agentIds);
				}
				$allAgentsName = trim($allAgentsName, ", ");
				//Get Production Job Data
				$productionJobData = $this->getProductJobData($jobId);
				$jobIdName = $productionJobData['job_id'];
				//Get Production Stage Data
				$productionStatusData = $this->getProductJobStageData($productionJobStageDataArr[0]['stages_id']);
				$stageName = $productionStatusData['status_name'];
				//Adding to production job log for job creation
				$type = ($allPostPutVars['is_group'] == 0) ? 'Agent' : 'Group';
				$logData = [
					'job_id' => $jobId,
					'title' => 'Agent re-assigned',
					'description' => $type . ' ' . $allAgentsName . ' re-assigned to job #' . $jobIdName . ' for stage ' . $stageName . '.',
					'user_type' => $allPostPutVars['user_type'],
					'user_id' => $allPostPutVars['user_id'],
					'created_date' => date_time(
						'today', [], 'string'
					)
				];
				$this->addingProductionLog($logData);
				$sendingEmailData = [
					'customer_id' => '',
					'job_id' => $jobId,
					'stages_id' => '',
					'is_group' => $allPostPutVars['is_group'],
					'agent_ids' => $agentIdsArr,
				];

				$jsonResponse = [
					'status' => 1,
					'email_data' => $sendingEmailData,
					'message' => message('Stage Assignee', 'updated'),
				];
			}

		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Get orders for create production job
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   01 Oct 2020
	 * @return json response
	 */
	public function getProductionOrderList($request, $response) {
		$getStoreDetails = get_store_details($request);
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Orders', 'error'),
		];
		$orderInit = new Orders();
		$orders = $orderInit->where('production_status', '0')
			->where('store_id', $getStoreDetails['store_id'])
			->orderBy('xe_id', 'DESC');
		$totalCount = $orders->count();
		if ($totalCount > 0) {
			$ordersData = $orders->get();
			$ordersData = json_clean_decode($ordersData, true);
			foreach ($ordersData as $orderKey => $storeOrders) {
				if ($storeOrders['order_number'] == '') {
					$ordersData[$orderKey]['order_number'] = $storeOrders['order_id'];
				}
			}
			$jsonResponse = [
				'status' => 1,
				'total' => $totalCount,
				'data' => $ordersData,
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * POST: Add internal note to production job
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   03 Oct 2020
	 * @return json response wheather data is saved or any error occured
	 */
	public function saveProductionJobInternalNote($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Note', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$jobId = to_int($allPostPutVars['job_id']);
		if ($jobId != '') {
			$productionJobInit = new ProductionJobs();
			$getOldJob = $productionJobInit->where('xe_id', $jobId);
			if ($getOldJob->count() > 0) {

				$allPostPutVars['created_date'] = date_time(
					'today', [], 'string'
				);
				$jobInternalNote = new ProductionJobNotes($allPostPutVars);
				if ($jobInternalNote->save()) {
					$noteInsertId = $jobInternalNote->xe_id;
					$allFileNames = do_upload(
						'upload',
						path('abs', 'production') . 'internal-note/', [150],
						'array'
					);
					//Save file name w.r.t note
					if (!empty($allFileNames)) {
						foreach ($allFileNames as $eachFile) {
							$fileData = [
								'note_id' => $noteInsertId,
								'file' => $eachFile,
							];
							$saveNoteFile = new ProductionJobNoteFiles($fileData);
							$saveNoteFile->save();
						}
					}
					$jsonResponse = [
						'status' => 1,
						'message' => message('Production Job Note', 'saved'),
					];
				}
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * POST: Add note to production job
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   03 Oct 2020
	 * @return json response wheather data is saved or any error occured
	 */
	public function saveProductionJobNote($request, $response) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Note', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$jobId = to_int($allPostPutVars['job_id']);
		if ($jobId != '') {
			$productionJobInit = new ProductionJobs();
			$getOldJob = $productionJobInit->where('xe_id', $jobId);
			if ($getOldJob->count() > 0) {
				$getOldJobData = $getOldJob->get();
				$getOldJobData = json_clean_decode($getOldJobData, true);
				$note = $getOldJobData[0]['note'];
				if ($allPostPutVars['user_type'] == 'agent') {
					$userInit = new User();
					$agent = $userInit->select('xe_id', 'name')->where('xe_id', $allPostPutVars['user_id']);
					if ($agent->count() > 0) {
                        $agentDetails = json_clean_decode($agent->first(), true);
                        $userName = $agentDetails['name'];
                    }
				} else {
					$userName = 'Admin';
				}
				if ($note == '') {
					$title = 'Note added by ' . $userName;
				} else {
					$title = 'Note updated by ' . $userName;
				}
				$productionJobInit->where('xe_id', $jobId)
					->update([
						'note' => $allPostPutVars['note'],
					]);
				//Adding to production job log
				$logData = [
					'job_id' => $jobId,
					'title' => $title,
					'description' => $allPostPutVars['note'],
					'user_type' => $allPostPutVars['user_type'],
					'user_id' => $allPostPutVars['user_id'],
					'created_date' => date_time(
						'today', [], 'string'
					)
				];
				$this->addingProductionLog($logData);
			}
			$jsonResponse = [
				'status' => 1,
				'message' => message('Production Job Note', 'updated'),
			];
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Send mail to agent when job is assigned
	 *
	 * @param $jobId  Production job id
	 * @param $isGroup  Is group
	 * @param $agentId  Agent Id
	 *
	 * @author debashrib@riaxe.com
	 * @date   03 Oct 2020
	 * @return array response
	 */

	private function sendEmailToAgent($request, $response, $jobId, $isGroup, $agentId, $getStoreDetails) 
	{
		$mailResponse = [];
		if ($agentId != '' && $agentId > 0 && $jobId != '' && $jobId > 0) {
			$productionJobInit = new ProductionJobs();

			$productionJob = $productionJobInit->where([
				'xe_id' => $jobId,
			]);
			if ($productionJob->count() > 0) {
				$productionJobArr = $productionJob->get();
				$productionJobArr = json_clean_decode($productionJobArr, true);
				$productionJobArr = $productionJobArr[0];
				$currentStage = $this->getJobCurrentStage($productionJobArr['xe_id']);
				$jobIdName = $productionJobArr['job_id'];
				$dueDate = date("M d, Y h:i A", strtotime($currentStage['exp_completion_date']));
				$link = API_URL . 'production-hub/production/' . $productionJobArr['xe_id'];
			}
			// Check the setting
			$settingData = $this->getProductionSetting($request, $response, ['module_id' => 4, 'return_type' => 1]);
			$settingData = $settingData['data'];
			if ($settingData['is_communication_enabled']) {
				//Get smtp email setting data for sending email
				$smtpSettingData = $this->getSmtpSettings(1, $getStoreDetails['store_id']? $getStoreDetails['store_id']:1);
				$emailData = $smtpSettingData['email_data'];
				$smtpData = $smtpSettingData['smtp_data'];
				$fromName = $emailData['from_email'];
				$fromEmail = $smtpData['smtp_from'];
				$replyToEmail = $emailData['to_email'];
				//Get all Agent list
				$userInit = new User();
            	$getUser = $userInit->select('admin_users.xe_id as id', 'admin_users.name', 'admin_users.email', 'user_role_rel.role_id')
                ->join('user_role_rel', 'admin_users.xe_id', '=', 'user_role_rel.user_id') 
                ->where('admin_users.store_id', $getStoreDetails['store_id']);
                $allAgentArr = json_clean_decode($getUser->get(), true);

				$subject = 'A new job #' . $jobIdName . ' assigned.';

				if ($isGroup == 0) {
					$agentDetails = array_filter($allAgentArr, function ($item) use ($agentId) {
						return $item['id'] == $agentId;
					});
					$agentDetails = $agentDetails[array_keys($agentDetails)[0]];
					$emailBody = '<span>Hello ' . $agentDetails['name'] . ',</span><br><br><span>A new job #' . $jobIdName . ' is assigned to you. The due date is ' . $dueDate . '.</span><br><span><a href="" target="' . $link . '">' . $link . '</a> to view the job details.</span><br><br><span>Thanks</span>';
					$mailContaint = ['from' => ['email' => $fromEmail, 'name' => $fromName],
						'recipients' => [
							'to' => [
								'email' => $agentDetails['email'],
								'name' => $agentDetails['name'],
							],
							'reply_to' => [
								'email' => $replyToEmail,
								'name' => $replyToEmail,
							],
						],
						'subject' => $subject,
						'body' => $emailBody,
						'smptData' => $smtpData,
					];
					$mailResponse = email($mailContaint);
				} else {
					$agentDetailsArr = array_filter($allAgentArr, function ($item) use ($agentId) {
						return $item['role_id'] == $agentId;
					});
					foreach ($agentDetailsArr as $agents) {
						$emailBody = '<span>Hello ' . $agents['name'] . ',</span><br><br><span>A new job #' . $jobIdName . ' is assigned to you. The due date is {due_date}.</span><br><span><a href="" target="' . $link . '">' . $link . '</a> to view the job details.</span><br><br><span>Thanks</span>';
						$mailContaint = ['from' => ['email' => $fromEmail, 'name' => $fromEmail],
							'recipients' => [
								'to' => [
									'email' => $agents['email'], //'debashrib@riaxe.com',//
									'name' => $agents['name'],
								],
								'reply_to' => [
									'email' => $replyToEmail,
									'name' => $replyToEmail,
								],
							],
							'subject' => $subject,
							'body' => $emailBody,
							'smptData' => $smtpData,
						];
						$mailResponse = email($mailContaint);
					}
				}
			}
		}
		return $mailResponse;
	}

	/**
	 * GET: Production Job Calender View
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   09 Oct 2020
	 * @return json response
	 */
	public function getProductionCalenderView($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$productionJobInit = new ProductionJobs();
		// Collect all Filter columns from url
		$agentId = $request->getQueryParam('agent_id');
		$from = $request->getQueryParam('from');
		$to = $request->getQueryParam('to');
		$agentIdArr = $request->getQueryParam('agent_id');
		$agentIdArr = json_clean_decode($agentIdArr, true);
		$printMethodArr = $request->getQueryParam('print_methods');
		$printMethodArr = json_clean_decode($printMethodArr, true);
		$stageId = $request->getQueryParam('stage_id');

		$productionJob = $productionJobInit
			->join('production_job_stages', 'production_jobs.current_stage_id', '=', 'production_job_stages.xe_id')
			->select('production_jobs.xe_id', 'production_jobs.job_id', 'production_jobs.order_id', 'production_jobs.order_item_id', 'production_jobs.order_item_quantity as quantity', 'production_jobs.job_title', 'production_jobs.current_stage_id', 'production_job_stages.stages_id', 'production_job_stages.stage_name', 'production_job_stages.stage_color_code', 'production_job_stages.exp_completion_date', 'production_job_stages.status')
			->where('production_jobs.store_id', $getStoreDetails['store_id'])
			->where('production_job_stages.status', '!=', 'completed');

		if (isset($from) && isset($to) && $from != "" && $to != "") {
			$to = date('Y-m-d H:i:s', strtotime($to . ' +1 day'));
			$productionJob->where('production_job_stages.exp_completion_date', '>=', $from)
				->where('production_job_stages.exp_completion_date', '<=', $to);
		}

		//Filter by print method id
		if (isset($printMethodArr) && !empty($printMethodArr)) {
			$productionJob->whereIn('production_job_stages.print_method_id', $printMethodArr);
		}

		//Filter by current stage id
		if (isset($stageId) && $stageId > 0) {
			$productionJob->where('production_job_stages.stages_id', '=', $stageId);
		}

		$productionJob->orderBy('production_jobs.created_at', 'DESC');
		$getTotalPerFilters = $productionJob->count();
		//Get all Agent list
		$userInit = new User();
        $getUser = $userInit->select('admin_users.xe_id as id', 'admin_users.name', 'admin_users.email', 'user_role_rel.role_id')
            ->join('user_role_rel', 'admin_users.xe_id', '=', 'user_role_rel.user_id') 
            ->where('admin_users.store_id', $getStoreDetails['store_id']);
        $allAgentArr = json_clean_decode($getUser->get(), true);
		if ($getTotalPerFilters > 0) {
			$productionJobArr = $productionJob->get();
			$productionJobArr = json_clean_decode($productionJobArr, true);
			$finalProductionJob = [];
			foreach ($productionJobArr as $productionJobs) {
				$tempProductionJob = $productionJobs;
				//Get assignee
				$productionJobAgentInit = new ProductionJobAgents();
				$assigneeData = $productionJobAgentInit->where([
					'job_id' => $productionJobs['xe_id'],
					'job_stage_id' => $productionJobs['current_stage_id'],
				]);
				$finalAssignee = [];
				$finalIsGroup = [];
				if ($assigneeData->count() > 0) {
					$assigneeDataArr = $assigneeData->get();
					foreach ($assigneeDataArr as $assignee) {
						array_push($finalAssignee, $assignee['agent_id']);
						array_push($finalIsGroup, $assignee['is_group']);
					}
				}
				$tempProductionJob['is_group'] = $finalIsGroup[0];
				$tempProductionJob['assignee_data'] = $finalAssignee;
				$nextStageId = $this->getNextStageId($productionJobs['xe_id'], $productionJobs['current_stage_id']);
				$tempProductionJob['next_stage_id'] = ($nextStageId > 0) ? $nextStageId : '';
				$previousStageId = $this->getPreviousStageId($productionJobs['xe_id'], $productionJobs['current_stage_id']);
				$tempProductionJob['previous_stage_id'] = ($previousStageId > 0) ? $previousStageId : '';
				
				$groupAgentId = [];
				if ($finalIsGroup[0] == 1) {
					foreach ($finalAssignee as $assignee) {
						$agentDetailsArr = array_filter($allAgentArr, function ($item) use ($assignee) {
							return $item['role_id'] == $assignee;
						});
						foreach ($agentDetailsArr as $agents) {
							array_push($groupAgentId, $agents['id']);
						}
					}
				}
				//For agent view
				if (isset($agentIdArr) && !empty($agentIdArr)) {
					$usedIds = [];
					foreach ($agentIdArr as $agentId) {
						//This condition is when agent is assigned
						if ($finalIsGroup[0] == 0 && in_array($agentId, $finalAssignee) && !in_array($tempProductionJob['xe_id'], $usedIds)) {
							$tempProductionJob['is_agent_operate'] = 1;
							array_push($finalProductionJob, $tempProductionJob);
							$usedIds[count($usedIds)] = $tempProductionJob['xe_id'];
							//This condition is when group is assigned
						} elseif ($finalIsGroup[0] == 1 && in_array($agentId, $groupAgentId) && !in_array($tempProductionJob['xe_id'], $usedIds)) {
							$tempProductionJob['is_agent_operate'] = 1;
							array_push($finalProductionJob, $tempProductionJob);
							$usedIds[count($usedIds)] = $tempProductionJob['xe_id'];
						}
					}

				} else {
					$tempProductionJob['is_agent_operate'] = 0;
					array_push($finalProductionJob, $tempProductionJob);
				}

			}
			$jsonResponse = [
				'status' => 1,
				'data' => $finalProductionJob,
			];
		} else {
			$jsonResponse = [
				'status' => 1,
				'data' => [],
				'message' => message('Production Job', 'not_found'),
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Get the next stage id
	 *
	 * @param $jobId Production Job Id
	 * @param $currentStageId Production job stage id
	 *
	 * @author debashrib@riaxe.com
	 * @date   09 Oct 2020
	 * @return json response
	 */
	private function getNextStageId($jobId, $currentStageId) {
		$nextStageId = 0;
		if ($jobId != '' && $jobId > 0 && $currentStageId != '' && $currentStageId > 0) {
			//$productionJobInit = new ProductionJobs();
			$productionJobStageInit = new ProductionJobStages();
			$allStages = $productionJobStageInit->where('job_id', $jobId)->orderBy('xe_id', 'ASC');
			if ($allStages->count() > 0) {
				$allStageDataArr = $allStages->get();
				$allStageDataArr = json_clean_decode($allStageDataArr, true);
				$nextStageData = [];
				foreach ($allStageDataArr as $stageKey => $stageData) {
					if ($stageData['xe_id'] == $currentStageId) {
						$nextStageKey = $stageKey + 1;
					}
				}
				$nextStageData = $allStageDataArr[$nextStageKey];
				if (!empty($nextStageData)) {
					$nextStageId = $nextStageData['xe_id'];
				}
			}
		}
		return $nextStageId;
	}

	/**
	 * POST: Change exp completion date of stage
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   03 Oct 2020
	 * @return json response wheather data is saved or any error occured
	 */
	public function changeExpCompletionData($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Due Date', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$jobStageId = to_int($allPostPutVars['job_stage_id']);
		$expCompletionDate = $allPostPutVars['exp_completion_date'];
		$newTime = $allPostPutVars['time'];
		$isBulk = $allPostPutVars['is_bulk'];
		$orderId = $allPostPutVars['order_id'];
		if ($jobStageId != '' && $jobStageId > 0 && $expCompletionDate != '') {
			if ((isset($isBulk) && $isBulk == 1) && (isset($orderId) && $orderId > 0)) {
				// Update due date for multiple jobs
				$productionJobInit = new ProductionJobs();
				$productionStageIds = $productionJobInit->where([
					'store_id' => $allPostPutVars['store_id'],
					'order_id' => $orderId,
				])->select('current_stage_id','job_status')->get()->toArray();
				if (!empty($productionStageIds)) {
					$productionJobStageInit = new ProductionJobStages();
					foreach ($productionStageIds as $value) {
						// If job is completed then skip this.
						if ($value['job_status'] == "completed") {
							continue;
						}
						$jobStageId = $value['current_stage_id'];
						$getOldJobStage = $productionJobStageInit->where('xe_id', $jobStageId);
						if ($getOldJobStage->count() > 0) {
							$getOldJobStageData = $getOldJobStage->get();
							$getOldJobStageData = json_clean_decode($getOldJobStageData, true);
							$oldExpCompDate = $getOldJobStageData[0]['exp_completion_date'];
							$jobId = $getOldJobStageData[0]['job_id'];
							$printMethodId = $getOldJobStageData[0]['print_method_id'];
							$stageName = $getOldJobStageData[0]['stage_name'];
							$oldExpCompDateArr = explode(' ', $oldExpCompDate);
							$time = (isset($newTime) && $newTime != '') ? $newTime : $oldExpCompDateArr[1];
							$displayOldExpCompDate = date("M d, Y h:i A", strtotime($oldExpCompDate));
							$newExpCompDate = $expCompletionDate . ' ' . $time;
							$displayNewExpCompDate = date("M d, Y h:i A", strtotime($newExpCompDate));
							$currentDateTime = date_time(
								'today', [], 'string'
							);
							if ($newExpCompDate > $currentDateTime) {
								$status = 'in-progress';
							} elseif ($newExpCompDate < $currentDateTime) {
								$status = 'delay';
							} else {
								$status = '';
							}
							$productionJobStageInit->where('xe_id', $jobStageId)
								->update([
									'exp_completion_date' => $newExpCompDate,
									'status' => $status,
								]);

							//Fetch data after updating
							$currentData = $productionJobStageInit
								->join('production_jobs', 'production_job_stages.job_id', '=', 'production_jobs.xe_id')
								->select('production_job_stages.xe_id', 'production_job_stages.job_id', 'production_job_stages.print_method_id', 'production_job_stages.stages_id', 'production_job_stages.stage_name', 'production_job_stages.stage_color_code', 'production_job_stages.created_date', 'production_job_stages.starting_date', 'production_job_stages.exp_completion_date', 'production_job_stages.completion_date', 'production_job_stages.status', 'production_job_stages.message', 'production_jobs.job_id as job_id_name')
								->where('production_job_stages.xe_id', $jobStageId);
							$currentDataArr = $currentData->get();
							$currentDataArr = json_clean_decode($currentDataArr, true);

							$jobIdName = $currentDataArr[0]['job_id_name']; //$productionJobData['job_id'];
							//Get All Print Profile
							$printMethodData = $this->getPrintMethodData($printMethodId);
							$printMethodName = $printMethodData['name'];
							if ($printMethodName != '') {
								$description = 'Due date of job #' . $jobIdName . ' changed from ' . $displayOldExpCompDate . ' to ' . $displayNewExpCompDate . ' for print method ' . $printMethodName . ' of stage ' . $stageName . '.';
							} else {
								$description = 'Due date of job #' . $jobIdName . ' changed from ' . $displayOldExpCompDate . ' to ' . $displayNewExpCompDate . ' of stage ' . $stageName . '.';
							}
							$logData = [
								'job_id' => $jobId,
								'title' => 'Due date changed',
								'description' => $description,
								'user_type' => $allPostPutVars['user_type'],
								'user_id' => $allPostPutVars['user_id'],
								'created_date' => date_time(
									'today', [], 'string'
								)
							];
							$this->addingProductionLog($logData);
						}
					}
				}
			} else {
				$productionJobStageInit = new ProductionJobStages();
				$getOldJobStage = $productionJobStageInit->where('xe_id', $jobStageId);
				if ($getOldJobStage->count() > 0) {
					$getOldJobStageData = $getOldJobStage->get();
					$getOldJobStageData = json_clean_decode($getOldJobStageData, true);
					$oldExpCompDate = $getOldJobStageData[0]['exp_completion_date'];
					$jobId = $getOldJobStageData[0]['job_id'];
					$printMethodId = $getOldJobStageData[0]['print_method_id'];
					$stageName = $getOldJobStageData[0]['stage_name'];
					$oldExpCompDateArr = explode(' ', $oldExpCompDate);
					$time = (isset($newTime) && $newTime != '') ? $newTime : $oldExpCompDateArr[1];
					$displayOldExpCompDate = date("M d, Y h:i A", strtotime($oldExpCompDate));
					$newExpCompDate = $expCompletionDate . ' ' . $time;
					$displayNewExpCompDate = date("M d, Y h:i A", strtotime($newExpCompDate));
					$currentDateTime = date_time(
						'today', [], 'string'
					);
					if ($newExpCompDate > $currentDateTime) {
						$status = 'in-progress';
					} elseif ($newExpCompDate < $currentDateTime) {
						$status = 'delay';
					}
					$productionJobStageInit->where('xe_id', $jobStageId)
						->update([
							'exp_completion_date' => $newExpCompDate,
							'status' => $status,
						]);

					//Fetch data after updating
					$currentData = $productionJobStageInit
						->join('production_jobs', 'production_job_stages.job_id', '=', 'production_jobs.xe_id')
						->select('production_job_stages.xe_id', 'production_job_stages.job_id', 'production_job_stages.print_method_id', 'production_job_stages.stages_id', 'production_job_stages.stage_name', 'production_job_stages.stage_color_code', 'production_job_stages.created_date', 'production_job_stages.starting_date', 'production_job_stages.exp_completion_date', 'production_job_stages.completion_date', 'production_job_stages.status', 'production_job_stages.message', 'production_jobs.job_id as job_id_name')
						->where('production_job_stages.xe_id', $jobStageId);
					$currentDataArr = $currentData->get();
					$currentDataArr = json_clean_decode($currentDataArr, true);

					$jobIdName = $currentDataArr[0]['job_id_name']; //$productionJobData['job_id'];
					//Get All Print Profile
					$printMethodData = $this->getPrintMethodData($printMethodId);
					$printMethodName = $printMethodData['name'];
					if ($printMethodName != '') {
						$description = 'Due date of job #' . $jobIdName . ' changed from ' . $displayOldExpCompDate . ' to ' . $displayNewExpCompDate . ' for print method ' . $printMethodName . ' of stage ' . $stageName . '.';
					} else {
						$description = 'Due date of job #' . $jobIdName . ' changed from ' . $displayOldExpCompDate . ' to ' . $displayNewExpCompDate . ' of stage ' . $stageName . '.';
					}
					$logData = [
						'job_id' => $jobId,
						'title' => 'Due date changed',
						'description' => $description,
						'user_type' => $allPostPutVars['user_type'],
						'user_id' => $allPostPutVars['user_id'],
						'created_date' => date_time(
							'today', [], 'string'
						)
					];
					$this->addingProductionLog($logData);
				}
			}
			$jsonResponse = [
				'status' => 1,
				'data' => $currentDataArr[0],
				'message' => message('Due Date', 'updated'),
			];
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Get orders for which production job created
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   13 Oct 2020
	 * @return json response
	 */
	public function getProductionJobOrderList($request, $response) {
		$getStoreDetails = get_store_details($request);
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Orders', 'error'),
		];
		$orderInit = new Orders();
		$orders = $orderInit->where('store_id', $getStoreDetails['store_id'])
			->where('production_status', '!=', '0')
			->orderBy('xe_id', 'DESC');
		$totalCount = $orders->count();
		if ($totalCount > 0) {
			$ordersData = $orders->get();
			$jsonResponse = [
				'status' => 1,
				'total' => $totalCount,
				'data' => $ordersData,
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Get production job list
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   13 Oct 2020
	 * @return json response
	 */
	public function getProductionJobList($request, $response) {
		$getStoreDetails = get_store_details($request);
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job', 'error'),
		];
		$productionJobInit = new ProductionJobs();
		$productionJob = $productionJobInit->select('xe_id', 'job_id')->where('store_id', $getStoreDetails['store_id'])
			->orderBy('xe_id', 'DESC');
		$totalCount = $productionJob->count();
		if ($totalCount > 0) {
			$productionJobData = $productionJob->get();
			$jsonResponse = [
				'status' => 1,
				'total' => $totalCount,
				'data' => $productionJobData,
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Production Activity Log
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   26 Sept 2020
	 * @return json response
	 */
	public function getProductionActivityLog($request, $response) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$productionJobInit = new ProductionJobs();
		// Collect all Filter columns from url
		$orderIdArr = $request->getQueryParam('order_id');
		$orderIdArr = json_clean_decode($orderIdArr, true);
		$jobId = $request->getQueryParam('job_id');
		$stageId = $request->getQueryParam('stage_id');
		$agentIdArr = $request->getQueryParam('agent_id');
		$agentIdArr = json_clean_decode($agentIdArr, true);
		$printMethodArr = $request->getQueryParam('print_methods');
		$printMethodArr = json_clean_decode($printMethodArr, true);
		$isOverDue = $request->getQueryParam('is_over_due');
		$from = $request->getQueryParam('from');
		$to = $request->getQueryParam('to');
		$customerId = $request->getQueryParam('customer_id');

		$productionJob = $productionJobInit
			->join('production_job_stages', 'production_jobs.current_stage_id', '=', 'production_job_stages.xe_id')
			->select('production_jobs.xe_id', 'production_job_stages.stages_id')
			->where('production_jobs.store_id', $getStoreDetails['store_id']);

		//Filter by customer
		$customerOrderData = [];
		if (isset($customerId) && $customerId > 0) {
			$orderInit = new Orders();
			$orderData = $orderInit
				->select('order_id')
				->where('store_id', $getStoreDetails['store_id'])
				->where('production_status', '!=', '0')
				->where('customer_id', $customerId);
			if ($orderData->count() > 0) {
				$orderDataArr = $orderData->get();
				$orderDataArr = json_clean_decode($orderDataArr, true);
				foreach ($orderDataArr as $orders) {
					array_push($customerOrderData, $orders['order_id']);
				}
			}
		}
		if (empty($orderIdArr)) {
			$orderIdArr = $customerOrderData;
		} else {
			$orderIdArr = array_merge($customerOrderData, $orderIdArr);
		}
		//Filter by order
		if (isset($orderIdArr) && !empty($orderIdArr)) {
			$productionJob->whereIn('production_jobs.order_id', $orderIdArr);
		}

		//Filter by Production job
		if (isset($jobId) && $jobId > 0) {
			$productionJob->where('production_jobs.xe_id', $jobId);
		}

		//Filter by Production job current stage
		if (isset($stageId) && $stageId > 0) {
			$productionJob->where('production_job_stages.stages_id', $stageId);
		}

		//Filter by Production job current stage over due
		if (isset($isOverDue) && $isOverDue == 1) {
			$productionJob->where('production_job_stages.status', 'delay');
		}

		//Filter by print method id
		if (isset($printMethodArr) && !empty($printMethodArr)) {
			$productionJob->whereIn('production_job_stages.print_method_id', $printMethodArr);
		}

		//Get all Agent list
		$userInit = new User();
        $getUser = $userInit->select('admin_users.xe_id as id', 'admin_users.name', 'admin_users.email', 'user_role_rel.role_id')
            ->join('user_role_rel', 'admin_users.xe_id', '=', 'user_role_rel.user_id') 
            ->where('admin_users.store_id', $getStoreDetails['store_id']);
        $allAgentArr = json_clean_decode($getUser->get(), true);

		$getTotalPerFilters = $productionJob->count();
		if ($getTotalPerFilters > 0) {
			$productionJobs = $productionJob->get();
			$productionJobArr = json_clean_decode($productionJobs, true);
			$finalProductionJobIds = [];
			foreach ($productionJobArr as $productionJobs) {
				//Get assignee
				$productionJobAgentInit = new ProductionJobAgents();
				$assigneeData = $productionJobAgentInit->select('is_group', 'agent_id')->where([
					'job_id' => $productionJobs['xe_id'],
					'job_stage_id' => $productionJobs['current_stage_id'],
				]);
				$finalAssignee = [];
				if ($assigneeData->count() > 0) {
					$assigneeDataArr = $assigneeData->get();
					$assigneeDataArr = json_clean_decode($assigneeDataArr, true);
					foreach ($assigneeDataArr as $assignee) {
						array_push($finalAssignee, $assignee['agent_id']);
					}
				}
				$isGroup = $assigneeDataArr[0]['is_group'];
				$groupAgentId = [];
				if ($isGroup == 1) {
					foreach ($finalAssignee as $assignee) {
						$agentDetailsArr = array_filter($allAgentArr, function ($item) use ($assignee) {
							return $item['role_id'] == $assignee;
						});
						foreach ($agentDetailsArr as $agents) {
							array_push($groupAgentId, $agents['id']);
						}
					}
				}
				if (isset($agentIdArr) && !empty($agentIdArr)) {
					foreach ($agentIdArr as $agentId) {
						//This condition is when agent is assigned
						if ($isGroup == 0 && in_array($agentId, $finalAssignee)) {
							array_push($finalProductionJobIds, $productionJobs['xe_id']);
							//This condition is when group is assigned
						} elseif ($isGroup == 1 && in_array($agentId, $groupAgentId)) {
							array_push($finalProductionJobIds, $productionJobs['xe_id']);
						}
					}
				} else {
					array_push($finalProductionJobIds, $productionJobs['xe_id']);
				}

			}
			//Get production log
			$productionJobLogInit = new ProductionJobLog();

			$logData = $productionJobLogInit->whereIn('job_id', $finalProductionJobIds);

			if (isset($from) && isset($to) && $from != "" && $to != "") {
				$to = date('Y-m-d H:i:s', strtotime($to . ' +1 day'));
				$logData->where('created_date', '>=', $from)
					->where('created_date', '<=', $to);
			}
			$logData->orderBy('created_date', 'DESC');
			$finalLogData = [];
			if ($logData->count() > 0) {
				$logDataArr = $logData->get();
				foreach ($logDataArr as $logs) {
					$tempLogData = $logs;
					$userName = $logs['user_type'];
					if ($logs['user_type'] == 'agent') {
						//Get agent name
						$userInit = new User();
						$agent = $userInit->select('xe_id', 'name')->where('xe_id', $logs['user_id']);
						if ($agent->count() > 0) {
                            $agentDetails = json_clean_decode($agent->first(), true);
                            $userName = $agentDetails['name'];
                        }
					}
					$tempLogData['user_name'] = $userName;
					$tempLogData['title'] = stripslashes($logs['title']);
					$tempLogData['description'] = stripslashes($logs['description']);
					$tempLogData['created_at'] = $logs['created_date'];
					$tempLogData['log_type'] = 'job_log';
					unset(
						$logs['created_date']
					);
					array_push($finalLogData, $tempLogData);
				}

			}

			//Get internal note data
			$internalNoteInit = new ProductionJobNotes();
			$internalNotes = $internalNoteInit->with('files')->whereIn('job_id', $finalProductionJobIds);
			if (isset($from) && isset($to) && $from != "" && $to != "") {
				$to = date('Y-m-d H:i:s', strtotime($to . ' +1 day'));
				$internalNotes->where('created_date', '>=', $from)
					->where('created_date', '<=', $to);
			}
			$internalNotes->orderBy('created_date', 'DESC');
			$noteRes = [];
			if ($internalNotes->count() > 0) {
				$noteDataArr = $internalNotes->get();
				foreach ($noteDataArr as $noteData) {
					$newNoteArr = $noteData;
					$userName = $newNoteArr['user_type'];
					if ($newNoteArr['user_type'] == 'agent') {
						//Get agent name
						$userInit = new User();
						$agent = $userInit->select('xe_id', 'name')->where('xe_id', $newNoteArr['user_id']);
						if ($agent->count() > 0) {
                            $agentDetails = json_clean_decode($agent->first(), true);
                            $userName = $agentDetails['name'];
                        }
					}
					$newNoteArr['title'] = 'Internal note added by ' . $userName;
					$newNoteArr['description'] = $newNoteArr['note'];
					$newNoteArr['log_type'] = 'internal_note';
					$newNoteArr['user_name'] = $userName;
					$newNoteArr['created_at'] = $newNoteArr['created_date'];
					unset(
						$newNoteArr['note'],
						$newNoteArr['seen_flag'],
						$newNoteArr['created_date']
					);
					array_push($noteRes, $newNoteArr);
				}
			}
			$totalProductionJobLogs = array_merge($finalLogData, $noteRes);
			// Sort the array by Created Date and time
			usort($totalProductionJobLogs, 'date_compare');
			if (is_array($totalProductionJobLogs) && !empty($totalProductionJobLogs)) {
				$jsonResponse = [
					'status' => 1,
					'data' => $totalProductionJobLogs,
				];
			}
		} else {
			$jsonResponse = [
				'status' => 0,
				'message' => message('Production Job', 'not_found'),
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Send email to customer after completion of stage
	 *
	 * @param $customerId Customer Id
	 * @param $jobId Production Job Id
	 * @param $stageId Production job stage id
	 *
	 * @author debashrib@riaxe.com
	 * @date   19 Oct 2020
	 * @return json response
	 */
	private function sendEmailToCustomer($request, $customerId, $jobId, $stageId) 
	{
		$getStoreDetails = get_store_details($request);
		$mailResponse = [];
		if ($customerId != '' && $jobId > 0 && $stageId > 0) {

			//Get production job details
			$productionJobInit = new ProductionJobs();
			$productionJob = $productionJobInit
				->join('orders', 'production_jobs.order_id', '=', 'orders.order_id')
				->select('production_jobs.xe_id', 'production_jobs.store_id', 'production_jobs.job_id', 'production_jobs.order_id', 'production_jobs.order_item_id', 'production_jobs.order_item_quantity', 'production_jobs.job_title', 'production_jobs.job_status', 'production_jobs.note', 'production_jobs.comp_percentage', 'production_jobs.due_date', 'production_jobs.scheduled_date', 'production_jobs.created_at', 'production_jobs.current_stage_id', 'orders.customer_id')
				->where([
					'production_jobs.xe_id' => $jobId,
					'production_jobs.store_id' => $getStoreDetails['store_id'],
				]);
			$productionJobData = $productionJob->get();
			$productionJobData = json_clean_decode($productionJobData, true);

			//Get customer data
			if ($customerId > 0) {
				$customersControllerInit = new CustomersController();
				$customerDetails = $customersControllerInit->getQuoteCustomerDetails($customerId, $getStoreDetails['store_id'], '');
				if (!empty($customerDetails)) {
					$customerName = ($customerDetails['customer']['name'] != '') ? $customerDetails['customer']['name'] : $customerDetails['customer']['email'];
			 		$customerEmail = $customerDetails['customer']['email'];
				}
			} elseif ($customerId == 0) {
				$orderId = $productionJobData[0]['order_id'];
				$ordersControllerInit = new OrdersController();
            	$orderDetails = $ordersControllerInit->getOrderDetails($request, $response, ['id' => $orderId], 1);
				$orderDetails = $orderDetails['data'];
				$customerName = ($orderDetails['customer_first_name'] != '') ? $orderDetails['customer_first_name'] . ' ' . $orderDetails['customer_last_name'] : $orderDetails['customer_email'];
				$customerEmail = $orderDetails['customer_email'];
			}
			//Get stage data
			$productionStatuInit = new ProductionStatus();
			$stageData = $productionStatuInit->where('xe_id', $stageId);
			if ($stageData->count() > 0) {
				$stageDataArr = $stageData->get();
				$stageDataArr = json_clean_decode($stageDataArr, true);
				$statusName = $stageDataArr[0]['status_name'];
				$templateTypeName = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '_', $statusName)));
				$templateInit = new ProductionEmailTemplates();
				$tempDataArr = $templateInit
					->where(
						[
							'template_type_name' => $templateTypeName,
							'module_id' => 4,
							'store_id' => $getStoreDetails['store_id'],
						]
					);
				if ($tempDataArr->count() > 0) {
					$templateData = $tempDataArr->get();
					$templateData = json_clean_decode($templateData, true);
					$isConfigured = $templateData[0]['is_configured'];
					$subject = $templateData[0]['subject'];
					$message = $templateData[0]['message'];
					if ($isConfigured == 1 && $subject != '' && $message != '') {
						$abbrivationsInit = new ProductionAbbriviations();
						$getAbbrivations = $abbrivationsInit->where('module_id', 4);
						$getAbbrivations = $getAbbrivations->get();
						$abbriviationData = json_clean_decode($getAbbrivations, true);
						foreach ($abbriviationData as $abbrData) {
							$abbrValue = $this->getAbbriviationValue($abbrData['abbr_name'], $productionJobData[0], $getStoreDetails);
							if (strpos($message, $abbrData['abbr_name']) !== false) {
								$message = str_replace($abbrData['abbr_name'], $abbrValue, $message);
							}
							if (strpos($subject, $abbrData['abbr_name']) !== false) {
								$subject = str_replace($abbrData['abbr_name'], $abbrValue, $subject);
							}
						}
						//Get smtp email setting data for sending email
						$smtpSettingData = $this->getSmtpSettings(1, $getStoreDetails['store_id']? $getStoreDetails['store_id']:1);
						$emailData = $smtpSettingData['email_data'];
						$smtpData = $smtpSettingData['smtp_data'];
						$fromName = $emailData['from_email'];
						$fromEmail = $smtpData['smtp_from'];
						$replyToEmail = $emailData['to_email'];
						$mailContaint = ['from' => ['email' => $fromEmail, 'name' => $fromName],
							'recipients' => [
								'to' => [
									'email' => $customerEmail,
									'name' => $customerName,
								],
								'reply_to' => [
									'email' => $replyToEmail,
									'name' => $replyToEmail,
								],
							],
							'attachments' => [],
							'subject' => $subject,
							'body' => $message,
							'smptData' => $smtpData,
						];
						$mailResponse = email($mailContaint);
					}
				}
			}
		}
		return $mailResponse;
	}

	/**
	 * Get Email Template Abbriviation Value
	 *
	 * @param $abbrName  Abbriviation Name
	 * @param $productionJobData Production job data array
	 *
	 * @author debashrib@riaxe.com
	 * @date   19 Oct 2020
	 * @return array response
	 */

	private function getAbbriviationValue($abbrName, $productionJobData, $getStoreDetails) 
	{
		//Get Customer Details
		$customersControllerInit = new CustomersController();
		$customerDetails = $customersControllerInit->getQuoteCustomerDetails($productionJobData['customer_id'], $getStoreDetails['store_id'], '');
		if (!empty($customerDetails)) {
			$customerName = ($customerDetails['customer']['name'] != '') ? $customerDetails['customer']['name'] : '';
	 		$customerEmail = ($customerDetails['customer']['email'] != '' ) ? $customerDetails['customer']['email'] : '';
		}
		//Get production job current stage
		$currentStage = $this->getJobCurrentStage($productionJobData['xe_id']);
		$printMethodId = $currentStage['print_method_id'];
		$printMethodName = '';
		if ($printMethodId > 0) {
			$printMethodData = $this->getPrintMethodData($printMethodId);
			$printMethodName = $printMethodData['name'];
		}
		$abbrValue = '';
		if ($abbrName == '{job_id}') {
			$abbrValue = $productionJobData['job_id'];
		} elseif ($abbrName == '{order_id}') {
			$abbrValue = $productionJobData['order_id'];
		} elseif ($abbrName == '{customer_name}') {
			$abbrValue = $customerName;
		} elseif ($abbrName == '{customer_email}') {
			$abbrValue = $customerEmail;
		} elseif ($abbrName == '{item_name}') {
			$abbrValue = $productionJobData['job_title'];
		} elseif ($abbrName == '{order_item_id}') {
			$abbrValue = $productionJobData['order_item_id'];
		} elseif ($abbrName == '{print_profile}') {
			$abbrValue = $printMethodName;
		} elseif ($abbrName == '{stage_name}') {
			$abbrValue = $currentStage['status_name'];
		}
		return $abbrValue;
	}

	/**
	 * POST: Sending email to customer and agents
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   27 Oct 2020
	 * @return json response wheather data is saved or any error occured
	 */
	public function sendEmail($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Sending Email', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$allPostPutVars = $request->getParsedBody();
		$emailDataArr = $allPostPutVars['email_data'];
		$emailDataArr = json_clean_decode($emailDataArr);
		if (!empty($emailDataArr)) {
			foreach ($emailDataArr as $emailData) {
				if ((string)$emailData['customer_id'] != ''
					&& (string)$emailData['stages_id'] != ''
					&& $emailData['job_id'] > 0) {
					//Sending mail to customer
					$this->sendEmailToCustomer($request, $emailData['customer_id'], $emailData['job_id'], $emailData['stages_id']);
				}

				if ($emailData['job_id'] > 0
					&& !empty($emailData['agent_ids'])) {
					foreach ($emailData['agent_ids'] as $agentId) {
						$this->sendEmailToAgent($request, $response, $emailData['job_id'], $emailData['is_group'], $agentId, $getStoreDetails);
					}
				}
			}
			$jsonResponse = [
				'status' => 1,
				'message' => message('Sending Email', 'done'),
			];
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}


	/**
	 * GET: Download Production Job as in pdf formate
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   03 June 2020
	 * @return json response
	 */
	public function downloadProductionJob($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Download', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$currency = '';
		$showPrice = $request->getQueryParam('show_price');
		if ($showPrice == 1) {
			//Get currency from global setting
	        $globalSettingData = $this->readSettingJsonFile($getStoreDetails['store_id']);
	        $currency = $globalSettingData['currency']['unicode_character'];
	    }
		if (!empty($args['id'])) {
			$settingData = $this->getProductionSetting($request, $response, ['module_id' => 4, 'return_type' => 1]);
			$settingData = $settingData['data'];
			$getProductionJobDetails = $this->getProductionJobDetails($request, $response, $args, 1);
		 	$dir = $this->createProductionJobPdf($getProductionJobDetails, $settingData, $showPrice, $currency, $getStoreDetails['store_id']);

		 	if (file_exists($dir['dir'])) {
                //Download file in local system
                if (file_download($dir['dir'])) {
                	if ($dir['qrcode_path'] != '') {
                		unlink($dir['qrcode_path']);
                	}
                    $serverStatusCode = OPERATION_OKAY;
                    $jsonResponse = [
                        'status' => 1,
                    ];
                }
            }
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}


	 /**
     * Create Production Job pdf
     *
     * @param $getProductionJobDetails     Production job details
     *
     * @author debashrib@riaxe.com
     * @date   06 June 2019
     * @return pdf file path
     */
    public function createProductionJobPdf($getProductionJobDetails, $settingData, $showPrice = 0, $currency = '', $storeId=1)
    {
    	if (!empty($getProductionJobDetails)) {
    		$ordersInit = new OrdersController();
        	$pdfLanguage = $ordersInit->languageContent['production-pdf'];
    		$productionJobData = $getProductionJobDetails['production_job'];
    		$orderData = $getProductionJobDetails['order_data'];
    		$barcode = generate_barcode($productionJobData['job_id']);
			$barcodeImageSrc = 'data:image/png;base64,' . base64_encode($barcode);

			$isBarcodeEnable = $settingData['is_barcode_enable'];
			$isProDescEnable = $settingData['is_show_product_description'];
			$timeFormate = $settingData['time_format'];
			$createdDate = date("M d,Y h:i a", strtotime($productionJobData['created_at']));
			$dueDate = date("M d,Y h:i a", strtotime($productionJobData['current_stage']['exp_completion_date']));
			if ($timeFormate == 24) {
				$createdDate = date("M d,Y H:i", strtotime($productionJobData['created_at']));
				$dueDate = date("M d,Y H:i", strtotime($productionJobData['current_stage']['exp_completion_date']));
			}
			$file = '';
			if ($isBarcodeEnable) {
				$token = 'job_id=' . $productionJobData['xe_id'].'&current_stage_id='.$productionJobData['current_stage']['xe_id'].'&store_id='.$productionJobData['store_id'];
	            $token = base64_encode($token);
	            $url = 'quotation/production-job?token=' . $token;
	            $url = API_URL . $url;
	            $uniqid = uniqid();
	            $file = path('abs', 'production').$uniqid.".png";
	            $showFile = path('read', 'production').$uniqid.".png";
	            $text = $url;
	            $ecc = 'L';
				$pixel_Size = 10;
				generate_qrcode($text, $file, $ecc, $pixel_Size, $frame_size);
			}
			/** get font for default language */
			$languageFont =  $this->getDefaultLanguageFont($storeId, "admin");
			// Fetch language Type
			$languageInit = new Language();
	        $where = ['type' => 'admin', 'is_default' => 1, 'store_id'=> $storeId];
	        $getLanguage = $languageInit->where($where)->first()->toArray();
			// End
			if ($getLanguage['name'] == "Japanese" || $getLanguage['name'] == "Chinese") {
				$fontFamily = "simsun";
				$fontAll = "simsun";
				$fontUrl = "";
			} else {
				$fontFamily = "DejaVuSans";
				$fontAll = "DejaVuSans";
				$fontUrl = "";
			}
	        $fontStyle = $languageFont['font_style'];
	        $fontWeight = $languageFont['font_weight'];
	        $trueType = $languageFont['true_type'];
			$class = (isset($languageFont['style']) && $languageFont['style']) == 'rtl' ? 'rtl' : '';

	        $rtlLeftAlign = "right";
	        $rtlRightAlign = "left";
	 		if ($getLanguage['name'] == 'Arabic') {
	 			$rtlLeftAlign = "left";
	        	$rtlRightAlign = "right";
	 		}

    		$html = '<!doctype html>
		        <html lang="en-US">
	          		<head>
	             		<meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
		                <style>@media print {
	                            body {-webkit-print-color-adjust: exact;}
	                        }
	                        @font-face {
								font-family: "'.$fontFamily.'";
								font-style: '.$fontStyle.';
								font-weight: '.$fontWeight.';
								src: url('.$fontUrl.') format("'.$trueType.'");
	                        }
	                        * {
								font-family: '.$fontAll.';
	                        }
	                    </style>
		          	</head>
		          	<body style="margin: 0; padding: 0;">
                <table cellpadding="5">

                <table width="100%" cellspacing="0" cellpadding="0" border="0" style="min-width: 100%;">
                <tr>
                	<td><br/>
                		<span style="margin: 0 0 0 auto; width: 120px;">
	                		<img src="@' . preg_replace('#^data:image/[^;]+;base64,#', '', $barcodeImageSrc) . '" style="width: auto; height:25px;" />
	                	</span>
	                </td>
                </tr>
                </table>
                <table width="100%" cellspacing="0" cellpadding="0" border="0" style="min-width: 100%;">
              <tr>
                <td width="70%" style="vertical-align: top;"><br/>
                  <strong style="font-size:24px;">'.$pdfLanguage['job'].'</strong><br/><br/>
                  <table width="100%" cellspacing="0" cellpadding="0" border="0" style="font-size: 14px;">
                    <tr>
                      <td width="30%" style="padding: 0 20px 4px 0px;">'.$pdfLanguage['job-id'].'</td>
                      <td width="70%" style="padding: 0 20px 4px 0px;">
                        : <strong>#' . $productionJobData['job_id'] . '</strong>
                      </td>
                    </tr>
                    <tr>
                      <td style="padding: 0 20px 4px 0px;">'.$pdfLanguage['job-title'].'</td>
                      <td style="padding: 0 20px 4px 0px;">
                        : <strong>' . $productionJobData['job_title'] . '</strong>
                      </td>
                    </tr>
                    <tr>
                      <td style="padding: 0 20px 4px 0px;">'.$pdfLanguage['created-date'].'</td>
                      <td style="padding: 0 20px 4px 0px;">
                        : <strong>' . $createdDate . '</strong>
                      </td>
                    </tr>
                    <tr>
                      <td style="padding: 0 20px 4px 0px;">'.$pdfLanguage['due-date'].'</td>
                      <td style="padding: 0 20px 4px 0px;">
                        : <strong>' . $dueDate . '</strong>
                      </td>
                    </tr>
                    <tr>
                      <td style="padding: 0 20px 4px 0px;">'.$pdfLanguage['stage'].'</td>
                      <td style="padding: 0 20px 4px 0px;">
                        : <strong>' . $productionJobData['current_stage']['status_name'] . '</strong>
                      </td>
                    </tr>
                  </table>
                </td>
                <td width="30%" align="'.$rtlLeftAlign.'" style="vertical-align: top; font-size: 14px;">
                <table width="100%" cellspacing="0" cellpadding="0" border="0" style="font-size: 14px;">';
                if ($isBarcodeEnable) {
                $html .= '<tr>
	                <td align="'.$rtlLeftAlign.'">
	                	<img src="' . $showFile . '" style="height:100%; max-height:100px; margin-top: 20px;">
	                </td>
                </tr>';
            	}
                $html .= '</table>
                </td>
              </tr>
              <tr>
              <td width="100%">
                <hr style="margin: 30px 0 0 0;">
              </td>
            </tr>
            </table>
            <table width="100%" cellspacing="0" cellpadding="0"  border="0" style="min-width: 100%; margin-bottom: 30px;">
              <tr>';
          	if ($orderData['customer_first_name'] != '') {
      			$customerName = $orderData['customer_first_name'] . ' ' . $orderData['customer_last_name'];
          	} else {
          		$customerName = $orderData['billing']['first_name'] . ' ' . $orderData['billing']['last_name'];
          	}
            $html .= '<td style="padding: 0 20px 4px 0px; font-size: 14px;"><span>'.$pdfLanguage['customer-name'].' : <strong>' . $customerName . '</strong></span></td>
              </tr>';	
          	if ($orderData['customer_email'] != '') {
            $html .= '<tr>
              	<td style="padding: 0 20px 4px 0px; font-size: 14px;"><span>'.$pdfLanguage['customer-email'].' : <strong>' . $orderData['customer_email'] . '</strong></span></td>
              </tr>';
          	}	
            $html .= '<tr>
                <td width="50%" style="vertical-align: top; font-size: 14px; line-height: 22px;"><br/>
                  <span style="font-size: 18px">'.$pdfLanguage['billing-address'].'</span>
                  <span><br/>
                    ' . $orderData['billing']['address_1'] . ', ' . $orderData['billing']['address_2'] . '<br/> 
                    ' . $orderData['billing']['city'] . ', ' . $orderData['billing']['state'] . '<br/> 
                    ' . $orderData['billing']['country'] . '-' . $orderData['billing']['postcode'] . '
                  </span>
                </td>
                <td width="50%" align="right" style="vertical-align: top; font-size: 14px; line-height: 22px;"><br/>
                  <span style="font-size: 18px">'.$pdfLanguage['shipping-address'].'</span><br/>';
				if ($orderData['shipping']['address_1'] != '') {
					$html .= '<span>
                    ' . $orderData['shipping']['address_1'] . ', ' . $orderData['shipping']['address_2'] . '<br/> 
                    ' . $orderData['shipping']['city'] . ', ' . $orderData['shipping']['state'] . '<br/> 
                    ' . $orderData['shipping']['country'] . '-' . $orderData['shipping']['postcode'] . '
                  </span>';
				}
				$html .= '</td>
              </tr>
            </table>';
            $display = ($productionJobData['note'] == '') ? 'display: none;' : '';
            $html .= '<table width="100%" cellspacing="0" cellpadding="0" border="0" style="margin-top: 30px;">
              <tr>
                <td>
                  <span style="' . $display . '">'.$pdfLanguage['note'].'</span>
                  <span style="font-size: 14px; line-height: 22px; ' . $display . '">
                    ' . $productionJobData['note'] . '
                  </span>
                </td>
              </tr>
            </table>';
				
			$html .= '<table width="100%" cellspacing="0" cellpadding="5" border="0">
              <tr>
                <td colspan="2">
                	<br/>
                  	<strong style="font-size:20px;" class="title mb-4">'.$pdfLanguage['order-id'].' #' . $orderData['order_number'] . '</strong>
                </td>
              </tr>
              <tr>
                <td width="20%" style="vertical-align: top;">
                  <table width="100%" cellspacing="0" cellpadding="0" border="0">
                    <tr>
                      <td style="padding: 5px;">
                        <span style="width: 100px; margin: 0;">
                          <img src="' . $getProductionJobDetails['product_image'][0]['thumbnail'] . '" style="width: auto; height:150px" alt=""/>
                        </span>
                      </td>
                    </tr>
                  </table>
                </td>
                <td width="80%" style="vertical-align: top; padding-left: 40px;">
                  <table width="100%" cellspacing="0" cellpadding="0" border="0" style="font-size: 14px;">
                    <tr>
                      <td width="30%" style="padding: 0 20px 4px 0px;">'.$pdfLanguage['product-name'].' </td>
                      <td width="70%" style="padding: 0 20px 4px 0px;">
                        <strong> ' . $orderData['orders']['name'] . '</strong>
                      </td>
                    </tr>
                     <tr>
                      <td width="30%" style="padding: 0 20px 4px 0px;">'.$pdfLanguage['sku'].' </td>
                      <td width="70%" style="padding: 0 20px 4px 0px;">
                        <strong> ' . $orderData['orders']['sku'] . '</strong>
                      </td>
                    </tr>
                     <tr>
                      <td width="30%" style="padding: 0 20px 4px 0px;">'.$pdfLanguage['quantity'].' </td>
                      <td width="70%" style="padding: 0 20px 4px 0px;">
                        <strong> ' . $orderData['orders']['quantity'] . '</strong>
                      </td>
                    </tr>';
                    // echo $currency;exit;
                    if ($showPrice == 1) {
                		$itemPrice = $orderData['orders']['price'];
                    	$html .= '<tr>
	                      <td width="30%" style="padding: 0 20px 4px 0px;">'.$pdfLanguage['price'].' </td>
	                      <td width="70%" style="padding: 0 20px 4px 0px;">
	                        <strong> <span style="font-family: '.$fontAll.';">'.$currency.';</span> ' . $itemPrice . '</strong>
	                      </td>
	                    </tr>';
                    }
                    if ($isProDescEnable) {
	                    $html .= '<tr>
	                      <td width="30%" class="'.$class.'" style="padding: 0 20px 4px 0px;  vertical-align: top;">'.$pdfLanguage['product-description'].' </td>
	                      <td width="70%" style="padding: 0 20px 4px 0px;">
	                        <strong> ' . $orderData['orders']['description'] . '</strong>
	                      </td>
	                    </tr>';
                    } 
                $html .= '</table>
                </td>
              </tr>
              <tr>
                <td colspan="2">';
                if (!empty($orderData['orders']['decoration_settings_data'])){
                  $html .= '<strong style="font-size: 16px; margin-top: 20px;"><br/>
                    '.$pdfLanguage['artwork-used'].'
                  </strong><br/>
                  ';
                    foreach ($orderData['orders']['decoration_settings_data'] as $key => $decorationSettingsData) {
                        if (!empty($decorationSettingsData['decoration_data'])) {
                            $html .= '<table width="100%" cellspacing="0" cellpadding="0" border="0" style="border: 0px solid #e3e3e3; font-size: 14px;">
                    <tr>
                      <td style="padding: 12px; border: 1px solid #e3e3e3;">
                        ';
                            foreach ($decorationSettingsData['decoration_data'] as $decoKey => $decorationData) {
                                $html .= '<table width="100%" cellspacing="0" cellpadding="0" border="0">
                          <tr>
                            <td align="center" width="20%" style="vertical-align: top;">
                            <strong style"font-size:14px">'.$pdfLanguage['preview'].'</strong><br/>
                              <span style="margin: 0; width: 150px; height: 150px;" >
                                <img src="' . $orderData['orders']['images'][$key]['src'] . '" alt="" style="width: 150px; height: 150px;" />
                              </span>
                            </td>
                             <td width="30%" style="vertical-align: top;">
                             <strong style"font-size:14px">'.$pdfLanguage['design'].'</strong><br/>
                              <span style="margin: 0; width: 120px; height: 70px;">';
								$html .= '<img src="' . $decorationData['png_url'] . '" alt="" style="width: auto; height: 70px;" /></span>';
								if ($decorationData['design_width'] != '' && $decorationData['design_height'] != '') {
									$html .= '<p style="text-align: '.$rtlRightAlign.'; font-size:12px;"><strong>' . $decorationData['design_width'] . ' X ' . $decorationData['design_height'] . ' (W X H) ' . $decorationData['print_unit'] . '</strong> </p>';
								}
								$html .= '
                            </td>
                            <td width="50%" style="vertical-align: top;">
                              <table>
                                <tr>
                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['area-name'].'</td>
                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>' . $decorationData['decoration_name'] . '</strong></td>
                                </tr>';
                                if ($decorationData['print_area_name'] != '') {
	                                $html .= '<tr>
	                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['decoration-area'].'</td>
	                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>' . $decorationData['print_area_name'] . '</strong></td>
	                                </tr>';
	                            }
                                $html .= '<tr>
                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['print-method'].'</td>
                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>' . $decorationData['print_profile_name'] . '</strong></td>
                                </tr>
                                <tr>
                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['width'].'</td>
                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>' . $decorationData['print_area_width'] . ' ' . $decorationData['print_unit'] . '</strong></td>
                                </tr>
                                <tr>
                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['height'].'</td>
                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>' . $decorationData['print_area_height'] . ' ' . $decorationData['print_unit'] . '</strong></td>
                                </tr>';
                                if (isset($decorationData['artwork_coordinates']) && !empty($decorationData['artwork_coordinates'])) {
	                                $html .= '<tr>
	                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['artwork-location'].'</td>
	                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>X: '.number_format($decorationData['artwork_coordinates']['x'], 2).', Y: '.number_format($decorationData['artwork_coordinates']['y'], 2).' ' . $decorationData['print_unit'] .'</strong></td>
	                                </tr>';
	                            }
	                            if (($decorationData['x_location'] > 0) && ($decorationData['y_location'] > 0)) {
	                                $html .= '<tr>
	                                  <td width="40%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;">'.$pdfLanguage['decoration-location'].'</td>
	                                  <td width="60%" style="text-align: '.$rtlRightAlign.'; padding: 0 0px 4px 20px;"><strong>X: '.$decorationData['x_location'].', Y: '.$decorationData['y_location'].' ' . $decorationData['print_unit'] .'</strong></td>
	                                </tr>';
	                            }
                              $html .= '</table>
                            </td>
                          </tr>
                          </table>';
                            }
                            $html .= '</td></tr></table>';
                        }
                        if(!empty($decorationSettingsData['stickerInfo'])) {
                        	$stickerInfo = $decorationSettingsData['stickerInfo'];
                        $html .='<span style="text-align: '.$rtlRightAlign.'; font-size: 16px; margin-top: 20px;">Sticker information</span>
                        	<table width="100%" cellspacing="0" cellpadding="0" border="0" style="border: 0px solid #e3e3e3; font-size: 14px;">
                        	<tbody>
                                <td>
	                                <table width="100%" cellspacing="0" cellpadding="0" border="0">
		                                <tr>
		                                  <td style="padding: 0 0px 4px 20px;">Size</td>
		                                  <td style="padding: 0 0px 4px 20px;"><strong>'.$stickerInfo['width'].' x '. $stickerInfo['height'].' '.$decorationSettingsData['decoration_data'][0]['print_unit'].'</strong></td>
		                                </tr>
		                                <tr>
		                                  <td style="padding: 0 0px 4px 20px;">Cutline type</td>
		                                  <td style="padding: 0 0px 4px 20px;"><strong>'.ucfirst($stickerInfo['stickerType']).'</strong></td>
		                                </tr>
		                                <tr>
		                                  <td style="padding: 0 0px 4px 20px;">Sticker option</td>
		                                  <td style="padding: 0 0px 4px 20px;"><strong>'.ucfirst($stickerInfo['stickerOption']).'</strong></td>
		                                </tr>
	                                </table>
                                </td>
                                <td>
                                	<table>
	                                	<tr>
		                                  <td style="padding: 0 0px 4px 20px;">Material</td>
		                                  <td style="padding: 0 0px 4px 20px;"><strong>'.$stickerInfo['material'].'</strong></td>
		                                </tr>
		                                <tr>
		                                  <td style="padding: 0 0px 4px 20px;">Cutline</td>
		                                  <td style="padding: 0 0px 4px 20px;"><strong>'.$stickerInfo['cutline'].'</strong></td>
		                                </tr>';
		                                if (!empty($stickerInfo['sheetInfo'])) {
		                                	$sheetInfo = $stickerInfo['sheetInfo'];
		                                	$sheetKey = $decorationSettingsData['sheet_key'];
		                                	$sheetDetails = $sheetInfo[$sheetKey]['name'];
			                                $html .= '<tr>
			                                  <td style="padding: 0 0px 4px 20px;">Sheet info</td>
			                                  <td style="padding: 0 0px 4px 20px;"><strong>'.$sheetDetails.'</strong></td>
			                                </tr>';
			                            }
			                            if (!empty($stickerInfo['rollInfo'])) {
			                            	$rollInfo = $stickerInfo['rollInfo'];
			                                $html .= '<tr>
			                                  <td style="padding: 0 0px 4px 20px;">Roll info</td>
			                                  <td style="padding: 0 0px 4px 20px;"><strong>'.$rollInfo['name'].'</strong></td> 
			                                  <td><img src="'.$rollInfo['option']['file_name'].'" alt="" style="width: auto; height: 35px;"/>
			                                  </td>
			                                </tr>';
			                            }
	                                $html .= '</table>
                                </td>
                                </tbody>
                              </table>';
                    	}
                    }
                	}
                    $html .= '</td>
              </tr>
            </table>';
            $html .= '</table>
        	</body></html>';
        	$filePath = path('abs', 'production');
        	// echo $html;exit;
		 	$fileNames = create_tcpdf($html, $filePath, $productionJobData['job_id'], $getLanguage['name']);
            $dir = $filePath . $fileNames;

            if (file_exists($dir)) {
            	$return = [
            		'dir' => $dir,
            		'qrcode_path' => $file
            	];
            	return $return;
            }
    	}
    }

    /**
	 * calculate due date 
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   06 June 2021
	 * @return date
	 */
	private function calculateDueDate($request, $response, $startingDate, $stageDuration) 
	{
		$settingData = $this->getProductionSetting($request, $response, ['module_id' => 4, 'return_type' => 1]);
		$settingData = $settingData['data'];
	
		$startingHours = '00:01';
		$endingHours  = '23:59';
		$weekendArr = [];
		$holidayListArr = [];
		
		if (isset($settingData['working_hours']) && !empty($settingData['working_hours'])) {
			$workingHours = $settingData['working_hours'];
			$startingHours = ($settingData['time_format'] == '12') ? date("H:i:s", strtotime($workingHours['starts_at'])) : $workingHours['starts_at'];
			$endingHours = ($settingData['time_format'] == '12') ? date("H:i:s", strtotime($workingHours['ends_at'])) : $workingHours['ends_at'];
		}
		if (isset($settingData['weekends']) && !empty($settingData['weekends'])) {
			$weekendArr = $settingData['weekends'];
		}
		if (isset($settingData['holiday_list']) && !empty($settingData['holiday_list'])) {
			$holidayListArr = array_column($settingData['holiday_list'], 'date');
		}
		$currentDate = date('Y-m-d', strtotime($startingDate));

		$finalStartingDate = $this->checkDueDateForAllCond($startingDate, $holidayListArr, $weekendArr, $startingHours, $endingHours);
		$expDueDate = $this->getExpDueDate($finalStartingDate, $stageDuration, $holidayListArr, $weekendArr, $startingHours, $endingHours);
		$result = [
			'start_date' => $finalStartingDate,
			'due_date' => $expDueDate
		];
		return $result;
	}

	private function checkDueDateForAllCond($startingDate, $holidayList, $weekendArr, $startingHours, $endingHours) 
	{
		//Check if starting date is working day or not
		$startingDate = $this->checkForHolidayList($startingDate, $holidayList);
		//Check if starting date is come under weekends
		$startingDate = $this->checkForWeekend($startingDate, $weekendArr, $holidayList);
		//Check for working hour
		$startingDate = $this->checkForWorkingHours($startingDate, $holidayList, $weekendArr, $startingHours, $endingHours);
		return $startingDate;
	}

	private function checkForHolidayList($date, $holidayList) 
	{
		$checkDate = date('Y-m-d', strtotime($date));
		if (!in_array($checkDate, $holidayList)) {
	  		return $date; 
	  	} else {
	  		$newDate = date('Y-m-d', strtotime($date . ' +1 day'));
	  		return $this->checkForHolidayList($newDate, $holidayList);
	  	}
	} 

	private function checkForWeekend($date, $weekendArr, $holidayList) 
	{
		$currentDay = strtolower(date('D', strtotime($date)));
		if (!in_array($currentDay, $weekendArr)) {
	  		return $date; 
	  	} else {
	  		$newDate = date('Y-m-d', strtotime($date . ' +1 day'));
	  		$newDate = $this->checkForHolidayList($newDate, $holidayList);
	  		return $this->checkForWeekend($newDate, $weekendArr, $holidayList);
	  	}
	}

	private function checkForWorkingHours($date, $holidayList, $weekendArr, $startingHours, $endingHours) 
	{
		$onlyDate = date('Y-m-d', strtotime($date));
		$startingDateTime = $onlyDate.' '.$startingHours;
		$endingDateTime = $onlyDate.' '.$endingHours;
		if (($date >= $startingDateTime) && ($date <= $endingDateTime)) {
		    return $date; 
	    //Check if date is before starting Hours
		} elseif ($date < $startingDateTime) {
			return $startingDateTime;
		//Check if date is after ending Hours
		} elseif ($date > $endingDateTime) {
			$newDate = date('Y-m-d', strtotime($onlyDate . ' +1 day'));
			//Check if starting date is working day or not
			$newDate = $this->checkForHolidayList($newDate, $holidayList);
			//Check if starting date is come under weekends
			$newDate = $this->checkForWeekend($newDate, $weekendArr, $holidayList);
			$newDate = date('Y-m-d', strtotime($newDate));
			$newStartingDate = $newDate.' '.$startingHours;
			return $newStartingDate;
		}
	}

	private function getExpDueDate($startDate, $remainingHours, $holidayList, $weekendArr, $startingHours, $endingHours) 
	{
		$remainingHoursArr = explode('.', $remainingHours);
		$stageDurationInSec = $remainingHours * 3600;
		
		//todays work
		$startingDateOnly = date('Y-m-d', strtotime($startDate));
		$endTimeForDate = $startingDateOnly.' '.$endingHours;
		$workOnDayInsec = strtotime($endTimeForDate) - strtotime($startDate);
		$remainingHoursInSec = $stageDurationInSec - $workOnDayInsec;
		$remainingworkingHours = round(($remainingHoursInSec / 3600), 2);
		if ($remainingworkingHours == 0 || $remainingworkingHours < 0) {
			$hours = floor($stageDurationInSec / 3600);
			$minutes = floor(($stageDurationInSec/60) - ($hours * 60));
			$expDeleviryDate = date('Y-m-d H:i:s', strtotime('+' . $hours . ' hour +' . $minutes . ' minutes', strtotime($startDate)));
			return $expDeleviryDate;
		} else {
			$onlyDate = date('Y-m-d', strtotime($startDate));
			$newDate = date('Y-m-d', strtotime($onlyDate . ' +1 day'));
			$newDate = $this->checkDueDateForAllCond($newDate, $holidayList, $weekendArr, $startingHours, $endingHours);
			return $this->getExpDueDate($newDate, $remainingworkingHours, $holidayList, $weekendArr, $startingHours, $endingHours);
		}
	}

	/**
	 * GET: Production Job Holiday List 
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   10 June 2020
	 * @return json response
	 */
	public function getProductionHolidayList($request, $response) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 1,
			'data' => [],
		];
		$getStoreDetails = get_store_details($request);
		$productionJobHolidaysInit = new ProductionJobHolidays();
		$holidayList = $productionJobHolidaysInit->where('store_id', $getStoreDetails['store_id'])
			->orderBy('date', 'ASC');
		if ($holidayList->count() > 0) {
			$getHolidayListData = $holidayList->get();
			$jsonResponse = [
				'status' => 1,
				'data' => $getHolidayListData,
			];
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}


	/**
	 * POST: Save Production Job Holiday List 
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   10 June 2020
	 * @return json response
	 */
	public function saveProductionHolidayList($request, $response) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Holiday List', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$allPostPutVars = $request->getParsedBody();

		if ($allPostPutVars['date'] != '' && $allPostPutVars['holiday_name'] != '' && $allPostPutVars['day'] != '') {
			$productionJobHolidaysInit = new ProductionJobHolidays();
			$holidayList = $productionJobHolidaysInit->where([
				'store_id' => $getStoreDetails['store_id'],
				'date' => $allPostPutVars['date']
			]);
			if ($holidayList->count() == 0) {
				//Add holiday
				$saveData = [
					'store_id' => $getStoreDetails['store_id'],
					'holiday_name' => $allPostPutVars['holiday_name'],
					'day' => $allPostPutVars['day'],
					'date' => $allPostPutVars['date'],
				];
				$holidaySave = new ProductionJobHolidays($saveData);
				if ($holidaySave->save()) {
					$holidayList = $productionJobHolidaysInit->where('store_id', $getStoreDetails['store_id'])
						->orderBy('date', 'ASC');
					if ($holidayList->count() > 0) {
						$getHolidayListData = $holidayList->get();
					}
					$jsonResponse = [
						'status' => 1,
						'message' => message('Production Job Holiday List', 'saved'),
						'data' => $getHolidayListData
					];
				}
				
			} else {
				$jsonResponse = [
					'status' => 0,
					'message' => message('Production Job Holiday List', 'exist'),
				];
			}
		}
		
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * GET: Production Job Stage Progress
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   15 June 2021
	 * @return json response
	 */
	public function getProductionStageProgress($request, $response, $args) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Stage Progress', 'not_found'),
		];
		$getStoreDetails = get_store_details($request);
		if (!empty($args['id'])) {
			$productionJobId = $args['id'];
			$productionJobInit = new ProductionJobs();

			$productionJob = $productionJobInit->where([
				'store_id' => $getStoreDetails['store_id'],
				'xe_id' => $productionJobId,
			]);
			$userInit = new User();
			$allAgent = $userInit->select('xe_id as id', 'name')->where('store_id', $getStoreDetails['store_id']);
			$allAgentArr = json_clean_decode($allAgent->get(), true);
			$userRoleInit = new UserRole();
			$allGroup = $userRoleInit->select('xe_id as id', 'role_name')->where('store_id', $getStoreDetails['store_id']);
			$allGroupArr = json_clean_decode($allGroup->get(), true);
			$finalProductJobData = [];
			if ($productionJob->count() > 0) {
				$productionJobStagesInit = new ProductionJobStages();
				$stageData = $productionJobStagesInit->select('xe_id', 'stage_name', 'stage_color_code', 'starting_date', 'exp_completion_date', 'completion_date', 'status', 'print_method_id')->where('job_id', $productionJobId)->orderBy('xe_id', 'ASC');
				if ($stageData->count() > 0) {
					$stageDataArr = $stageData->get();
					foreach ($stageDataArr as $stageKey => $stageData) {
						//Get print method name
						$printMethodData = $this->getPrintMethodData($stageData['print_method_id']);
						$printMethodName = $printMethodData['name'];
						$stageDataArr[$stageKey]['print_method_name'] = $printMethodName;
						
						$productionJobAgentInit = new ProductionJobAgents();
						$assigneeData = $productionJobAgentInit->where([
							'job_id' => $productionJobId,
							'job_stage_id' => $stageData['xe_id'],
						]);
						$finalAssignee = [];
						$finalIsGroup = [];
						if ($assigneeData->count() > 0) {
							$assigneeDataArr = $assigneeData->get();
							foreach ($assigneeDataArr as $assignee) {
								if ($assignee['is_group'] == 0) {
									$agentId = $assignee['agent_id'];
									$agentDetails = array_filter($allAgentArr, function ($item) use ($agentId) {
										return $item['id'] == $agentId;
									});
									$agentDetails = $agentDetails[array_keys($agentDetails)[0]];
									$tempData['id'] = $agentId;
									$tempData['name'] = $agentDetails['name'];
								} else {
									$groupId = $assignee['agent_id'];
									$groupDetails = array_filter($allGroupArr, function ($item) use ($groupId) {
										return $item['id'] == $groupId;
									});
									$groupDetails = $groupDetails[array_keys($groupDetails)[0]];
									$tempData['id'] = $groupId;
									$tempData['name'] = $groupDetails['role_name'];
								}
								array_push($finalAssignee, $tempData);
								array_push($finalIsGroup, $assignee['is_group']);
							}
						}
						$stageDataArr[$stageKey]['agents'] = $finalAssignee;
					}
				}
				$jsonResponse = [
					'status' => 1,
					'data' => $stageDataArr,
				];
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * POST : Update Production holiday list
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   18 June 2020
	 * @return Updated Json Status
	 */
	public function updateProductionHolidayList($request, $response, $args) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Holiday List', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$getStoreDetails = get_store_details($request);
		if (!empty($args) && $args['id'] > 0) {
			$holidayId = to_int($args['id']);
			$getHolidayListData = [];
			$productionJobHolidaysInit = new ProductionJobHolidays();
			$holidayData = $productionJobHolidaysInit->where('xe_id', $holidayId);
			if ($holidayData->count() > 0) {
				$productionJobHolidaysInit->where([
					'xe_id' => $holidayId,
					'store_id' => $getStoreDetails['store_id'],
				])->update([
					'holiday_name' => $allPostPutVars['holiday_name'],
					'day' => $allPostPutVars['day'],
					'date' => $allPostPutVars['date'],
				]);
				$holidayList = $productionJobHolidaysInit->where('store_id', $getStoreDetails['store_id'])
					->orderBy('date', 'ASC');
				if ($holidayList->count() > 0) {
					$getHolidayListData = $holidayList->get();
				}
				$jsonResponse = [
					'status' => 1,
					'message' => message('Production Job Holiday List', 'updated'),
					'data' => $getHolidayListData
				];
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Delete : Delete Production holiday list
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   10 June 2020
	 * @return Delete Json Status
	 */
	public function deleteProductionHolidayList($request, $response, $args) 
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Holiday List', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		if (!empty($args) && $args['id'] > 0) {
			$holidayId = to_int($args['id']);
			$getHolidayListData = [];
			$productionJobHolidaysInit = new ProductionJobHolidays();
			$holidayData = $productionJobHolidaysInit->where('xe_id', $holidayId);
			if ($holidayData->count() > 0) {
				$productionJobHolidaysInit->where('xe_id', $holidayId)->delete();
				$holidayList = $productionJobHolidaysInit->where('store_id', $getStoreDetails['store_id'])
					->orderBy('date', 'ASC');
				if ($holidayList->count() > 0) {
					$getHolidayListData = $holidayList->get();
				}
				$jsonResponse = [
					'status' => 1,
					'message' => message('Production Job Holiday List', 'deleted'),
					'data' => $getHolidayListData
				];
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}
	/**
	 * POST : production job stage
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author debashrib@riaxe.com
	 * @date   18 June 2020
	 * @return Updated Json Status
	 */
	public function productionJobStage($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job Status', 'error'),
		];
		$getStoreDetails = get_store_details($request);
		$storeId = $getStoreDetails['store_id'] ? $getStoreDetails['store_id'] : 1;
		$moduleId = 4;
		$allPostPutVars = $request->getParsedBody();
		$barCode = $allPostPutVars['bar_code'];
		$jobId = $allPostPutVars['job_id'];
		$barCodesInit = new BarCodes();
		$getBarCodeData = $barCodesInit->where(['barcode_number' => $barCode , 'store_id' => $storeId , 'module_id' => $moduleId ]);
		if($getBarCodeData->count() > 0) {
			$getBarCode =  $getBarCodeData->first()->toArray();
			$barCodeSlug = $getBarCode['slug'];
			$productionStatusInit =  new ProductionStatus();
			$getProductionStatus =  $productionStatusInit->where(['slug' => $barCodeSlug , 'store_id' => $storeId ]);
			if($getProductionStatus->count() > 0) {
				$getProductionData = $getProductionStatus->first()->toArray();
				$statusName =  $getProductionData['status_name'];
				$currentStageId = $this->getCurrentStageId($jobId , $statusName, 1);
				$nextStageId =  $currentStageId + 1;
				$this->getCurrentStageStatus($jobId ,$statusName);
				$stagesIds =  $this->getAllJobStageByJobId($jobId);
				if($currentStageId > 0) {
					if(!empty($stagesIds)) {
						foreach($stagesIds as $stagesId) {
							if($stagesId['xe_id'] == $nextStageId) {
								$this->updateJobNextStage($request, $response,$nextStageId);
							} elseif($stagesId['xe_id'] <= $currentStageId) {
								$nextStageIdToBe = $stagesId['xe_id'] + 1;
								$arguments = ["current_stage_id" => $stagesId['xe_id'] , "type" => "completed", "next_stage_id" => $nextStageIdToBe, 'is_return' => 1 ];
								$stageResposne =  $this->productionJobStageOperations($request, $response, $arguments);
							}
						}
						$jsonResponse = $stageResposne;
					} else {
						$statusMessage = "job status " .  strtolower($statusName)." already completed.";
						$jsonResponse = ['status' => 0 , 'data' => [] , 'message' => ucfirst($statusMessage)];
					}
				} else {
					$statusMessage = "job status " .  strtolower($statusName)." already completed.";
					$jsonResponse = ['status' => 0 , 'data' => [] , 'message' => ucfirst($statusMessage)];
				}
				
			}

		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}
	/**
	 * GET : Current stage id
	 *
	 * @param $jobId
	 * @param $statusName
	 *
	 * @author soumyas@riaxe.com
	 * @date   04 May 2022
	 * @return int
	 */
	public function getCurrentStageId($jobId , $statusName, $isBarcode= 0) {
		$currentStageId = 0;
		$productionJobStagesInit =  new ProductionJobStages();
		$getProductionJobStages =  $productionJobStagesInit->where(['job_id' => $jobId , 'stage_name' =>$statusName]);
		if($getProductionJobStages->count() > 0) {
			if($isBarcode == 1) {
				$getProductionJobStagesData =  $getProductionJobStages->orderBy('xe_id', "ASC")->get()->toArray();
				foreach($getProductionJobStagesData as $value) {
					if($value['status'] !='completed') {
						$currentStageId = $value['xe_id'];
						break;
					}
				}
			} else {
				$getProductionJobStagesData =  $getProductionJobStages->first()->toArray();
				$currentStageId = $getProductionJobStagesData['xe_id'];
			}
		}
		return $currentStageId;
	}
	/**
	 * GET : All stage id
	 *
	 * @param $jobId
	 * *
	 * @author soumyas@riaxe.com
	 * @date   04 May 2022
	 * @return Array
	 */
	public function getAllJobStageByJobId($jobId) {
		$stagesIds = [];
		$productionJobStagesInit =  new ProductionJobStages();
		$getProductionJobStages =  $productionJobStagesInit->select('xe_id')->where('job_id', '=', $jobId)->where('status' , '!=', 'completed')->orderBy('xe_id','ASC');
		if($getProductionJobStages->count() > 0) {
			$getProductionJobStagesArray =  $getProductionJobStages->get()->toArray();
			$stagesIds = $getProductionJobStagesArray;
		}
		return $stagesIds;
	}
	/**
	 * POST : Update next stage
	 *
	 * @param $request
	 * @param $response
	 * @param $nextProductionJobStageId
	 *
	 * @author soumyas@riaxe.com
	 * @date   04 May 2022
	 * @return boolean 
	 */
	public function updateJobNextStage($request, $response, $nextProductionJobStageId) {
		$allPostPutVars = $request->getParsedBody();
		$userType = $allPostPutVars['user_type'];
		$userId = $allPostPutVars['user_id'];
		$isJobNextStage = false;
		$printMethodName = '';
		$jobIdName = '';
		$stageName= '';
		if ($nextProductionJobStageId > 0) {
			$productionJobStageInit = new ProductionJobStages();
			$nextStage = $productionJobStageInit->where('xe_id', $nextProductionJobStageId);
			$nextStageData = $nextStage->get();
			$nextStageData = json_clean_decode($nextStageData, true);
			$newStageName = $nextStageData[0]['stage_name'];
			//Start next job
			$assignee = [];
			$nextStageData = $productionJobStageInit->where('xe_id', $nextProductionJobStageId);
			if ($nextStageData->count() > 0) {
				$nextStageDetails = $nextStageData->get();
				$nextStageDetails = json_clean_decode($nextStageDetails, true);
				$nextProductionStatusData = $this->getProductJobStageData($nextStageDetails[0]['stages_id']);
				$nextDuration = $nextProductionStatusData['feature_data']['duration'];
				$nextStartingDate = date_time(
					'today', [], 'string'
				);
				$getStartNDuedate = $this->calculateDueDate($request, $response, $nextStartingDate, $nextDuration);
				//Start the next stage
				$productionJobStageInit->where('xe_id', $nextProductionJobStageId)
					->update([
						'starting_date' => $getStartNDuedate['start_date'],
						'exp_completion_date' =>  $getStartNDuedate['due_date'],
						'status' => 'in-progress',
					]);
				//Update the current stage in production
				$productionJobInit = new ProductionJobs();
				$productionJobInit->where('xe_id', $nextStageDetails[0]['job_id'])
					->update([
						'current_stage_id' => $nextProductionJobStageId,
					]);
				//Add assignee data
				$assignee = $this->saveAssigneeData($request, $nextStageDetails[0]['job_id'], $nextStageDetails[0]['stages_id'], $nextProductionJobStageId);
				//Adding to production job log
				if ($allPostPutVars['user_id'] == 1 && $allPostPutVars['user_type'] == 'admin') {
					$by = ' by Admin';
				} else {
					$userInit = new User();
					$agent = $userInit->select('xe_id', 'name')->where('xe_id', $allPostPutVars['user_id'])->first()->toArray();
					$by = ' by Agent '.$agent['name'];
				}
				if ($printMethodName != '') {
					if ($newStageName != '') {
						if (!empty($assignee)) {
							$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
							$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
						}
						$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . ' for print method ' . $printMethodName . $by.' and auto assigned to ' . $type . ' ' . $names . '.';
					} else {
						$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . ' for print method ' . $printMethodName . $by.'.';
					}
				} else {
					if ($newStageName != '') {
						if (!empty($assignee)) {
							$type = (isset($assignee['is_group']) && $assignee['is_group'] == 0) ? 'Agent' : 'Group';
							$names = (isset($assignee['names']) && $assignee['names'] != '') ? $assignee['names'] : '';
						}
						$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . $by.' and auto assigned to ' . $type . ' ' . $names . '.';
					} else {
						$description = 'Status of Job #' . $jobIdName . ' changed from ' . $stageName . ' to ' . $newStageName . $by.'.';
					}
				}
				$logData = [
					'job_id' => $nextProductionJobStageId, //$stageDataArr['job_id'],
					'title' => 'Status changed and auto assigned',
					'description' => $description,
					'user_type' => $userType,
					'user_id' => $userId,
					'created_date' => date_time(
						'today', [], 'string'
					)
				];
				$this->addingProductionLog($logData);
			}
			$isJobNextStage = true;
		}
		return $isJobNextStage;
	}
	/**
	 * GET : Current stage status
	 *
	 * @param $jobId
	 * @param $statusName
	 *
	 * @author soumyas@riaxe.com
	 * @date   04 May 2022
	 * @return int 
	 */
	public function getCurrentStageStatus($jobId,$statusName) {
		$isCompleted = 0;
		$productionJobStagesInit =  new ProductionJobStages();
		$getProductionJobStages =  $productionJobStagesInit->where('job_id', '=', $jobId)->where('stage_name' , '=', $statusName)->where('status' , '=' , 'completed');
		if($getProductionJobStages->count() > 0) {
			$isCompleted = 1;
		}
		return $isCompleted;
	}

	/**
	 * GET: Production Job list current stage and next stage for respective orders.
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author mike@imprintnext.com
	 * @date   07 Sept 2023
	 * @return json response
	 */
	public function getProductionJobListStage($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Production Job List', 'not_found'),
		];
		$getStoreDetails = get_store_details($request);
		if (!empty($args['id'])) {
			$productionJobId = $args['id'];
			$productionJobInit = new ProductionJobs();

			$productionJob = $productionJobInit->where([
				'store_id' => $getStoreDetails['store_id'],
				'xe_id' => $productionJobId,
			]);

			$finalProductJobData = [];
			if ($productionJob->count() > 0) {
				$productionJobArr = $productionJob->select('order_id')->first();
				$orderId = $productionJobArr->order_id;
				$productionJobList = $productionJobInit->where([
					'store_id' => $getStoreDetails['store_id'],
					'order_id' => $orderId,
				]);
				$productionJobListArr = $productionJobList->get()->toArray();
				if (!empty($productionJobListArr)) {
					$i = 0;
					foreach ($productionJobListArr as $value) {
						$nextStageId = 0;
						$finalProductJobData[$i]['current_stage_id'] = $value['current_stage_id'];
						$finalProductJobData[$i]['job_id'] = $value['xe_id'];
						$nextStageId = $this->getNextStageId($value['xe_id'], $value['current_stage_id']);
						$finalProductJobData[$i]['next_stage_id'] = ($nextStageId > 0) ? $nextStageId : '';
						$i++;
					}
				}
				$jsonResponse = [
					'status' => 1,
					'data' => $finalProductJobData,
				];
			}
		}
		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

	/**
	 * Get the previous stage id
	 *
	 * @param $jobId Production Job Id
	 * @param $currentStageId Production job stage id
	 *
	 * @author diana@imprintnext.com
	 * @date   04 Oct 2022
	 * @return json response
	 */
	private function getPreviousStageId($jobId, $currentStageId) {
		$previousStageId = 0;
		if ($jobId > 0 && $currentStageId > 0) {
			$productionJobStageInit = new ProductionJobStages();
			$allStages = $productionJobStageInit->where('job_id', $jobId)->orderBy('xe_id', 'ASC');
			if ($allStages->count() > 0) {
				$allStageDataArr = $allStages->get();
				$allStageDataArr = json_clean_decode($allStageDataArr, true);
				$previousStageData = [];
				foreach ($allStageDataArr as $stageKey => $stageData) {
					if ($stageData['xe_id'] == $currentStageId) {
						$previousStageKey = $stageKey - 1;
					}
				}
				$previousStageData = $allStageDataArr[$previousStageKey];
				if (!empty($previousStageData)) {
					$previousStageId = $previousStageData['xe_id'];
				}
			}
		}
		return $previousStageId;
	}

	/**
	 * POST: Undo mark as done Production job stage
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author diana@imprintnext.com
	 * @date   07 Oct 2022
	 * @return json response
	 */
	public function productionJobStageUndoOperations($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Stage Operation', 'error'),
		];
		$allPostPutVars = $request->getParsedBody();
		$productionJobStageId = isset($allPostPutVars['current_stage_id']) ? $allPostPutVars['current_stage_id'] : 0;
		$previousProductionJobStageId = (isset($allPostPutVars['previous_stage_id']) && $allPostPutVars['previous_stage_id'] > 0) ? $allPostPutVars['previous_stage_id'] : 0;
		if ($productionJobStageId && $previousProductionJobStageId) {
			$productionJobInit = new ProductionJobs();
			$productionJobStageInit = new ProductionJobStages();
			$orderInit = new Orders();
			$stageData = $productionJobStageInit->where([
				'xe_id' => $productionJobStageId,
			]);
			if ($stageData->count() > 0) {
				$stageDataArr = $stageData->first();
				$stageDataArr = json_clean_decode($stageDataArr, true);
				
				//Get Production Job Data
				$productionJobData = $this->getProductJobData($stageDataArr['job_id']);
				$orderId = $productionJobData['order_id'];
				$jobIdName = $productionJobData['job_id'];
				$customerId = $productionJobData['customer_id'];

				//Get Production Stage Data
				$productionStatusData = $this->getProductJobStageData($stageDataArr['stages_id']);
				$stageName = $productionStatusData['status_name'];
				//Get New stage details
				if ($previousProductionJobStageId > 0) {
					//Get previous stage data
					$previousStageData = $productionJobStageInit->where([
						'xe_id' => $previousProductionJobStageId,
					]);
					$previousStageDataArr = $previousStageData->first();
					$previousStageDataArr = json_clean_decode($previousStageDataArr, true);
					$previousProductionStatusData = $this->getProductJobStageData($previousStageDataArr['stages_id']);
					$previousStageName = $previousProductionStatusData['status_name'];
					$previousDuration = $previousProductionStatusData['feature_data']['duration'];
					$previousStartingDate = date_time(
						'today', [], 'string'
					);
					$getStartDuedate = $this->calculateDueDate($request, $response, $previousStartingDate, $previousDuration);

					//Add to production log
					if ($allPostPutVars['user_id'] == 1 && $allPostPutVars['user_type'] == 'admin') {
						$by = ' by Admin';
					} else {
						$userInit = new User();
						$agent = $userInit->select('xe_id', 'name')->where('xe_id', $allPostPutVars['user_id'])->first()->toArray();
						$by = ' by Agent '.$agent['name'];
					}

					//Delete the agent assigment of current stage id
					$productionJobAgentInit = new ProductionJobAgents();
					$productionJobAgentInit->where(['job_stage_id' => $productionJobStageId])->delete();
					if ($stageDataArr['status'] == 'completed') {
						//Update the status as in-progress for current stage id
						$productionJobStageInit->where('xe_id', $productionJobStageId)
						->update([
							'status' => 'in-progress','completed_by' => null,
						]);
						//Update the jsob status as progressing
						$productionJobInit->where('xe_id', $stageDataArr['job_id'])
						->update([
							'job_status' => 'progressing',
							'current_stage_id' => $productionJobStageId
						]);

						// Set dynamic message
						$logDesc = 'Status of Job #'.$jobIdName.' '.$stageName.' reverted back from Completed to In-progress via undo'.$by.'.';
					} else {
						//Update the status as not-started for current stage id
						$productionJobStageInit->where('xe_id', $productionJobStageId)
						->update([
							'status' => 'not-started'
						]);
						//Update the status as in-progress for previous stage id
						$productionJobStageInit->where('xe_id', $previousProductionJobStageId)
						->update([
							'status' => 'in-progress',
							'completion_date' => null,
							'completed_by' => null,
							'starting_date' => $getStartDuedate['start_date'],
							'exp_completion_date' =>  $getStartDuedate['due_date'],
						]);
						//Delete the agent of previous stage and assign again
						$productionJobAgentInit->where(['job_stage_id' => $previousProductionJobStageId])->delete();
						//Add assignee data
						$assignee = $this->saveAssigneeData($request, $previousStageDataArr['job_id'], $previousStageDataArr['stages_id'], $previousProductionJobStageId);
						//Update the jsob status as progressing
						$productionJobInit->where('xe_id', $stageDataArr['job_id'])
						->update([
							'job_status' => 'progressing',
							'current_stage_id' => $previousProductionJobStageId
						]);

						// Set dynamic message
						$logDesc = 'Status of Job #'.$jobIdName.' reverted back from '.$stageName.' to '.$previousStageName.' via undo'.$by.'.';
					}

					$getAllStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->where('production_job_stages.job_id', $stageDataArr['job_id']);
					$getAllStageData = $getAllStage->get()->toArray();
					$stageTotalTimeArr = array_column($getAllStageData, 'duration');
					$stageTotalTime = array_sum($stageTotalTimeArr);
					$completedStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->where('production_job_stages.status', 'completed')->where('production_job_stages.job_id', $stageDataArr['job_id']);

					$completedStageData = $completedStage->get()->toArray();
					$stageCompTimeArr = array_column($completedStageData, 'duration');
					$stageCompTime = array_sum($stageCompTimeArr);
					$percentage = ($stageCompTime / $stageTotalTime) * 100;
					$productionJobInit->where('xe_id', $stageDataArr['job_id'])
					->update([
						'comp_percentage' => round($percentage),
					]);	

					//Update the order's production_status as 1
					$orderInit->where('order_id', $orderId)
					->update([
						'production_status' => 1,
					]);
					//Update the order's production_percentage
					$getAllJob = $productionJobInit->select('xe_id')->where('order_id', $orderId);
					$getAllJobData = $getAllJob->get()->toArray();
					$jobIdArr = array_column($getAllJobData, 'xe_id');
					$getAllStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->whereIn('production_job_stages.job_id', $jobIdArr);
					$getAllStageData = $getAllStage->get()->toArray();
					$stageTotalTimeArr = array_column($getAllStageData, 'duration');
					$stageTotalTime = array_sum($stageTotalTimeArr);
					$completedStage = $productionJobStageInit->select('production_job_stages.xe_id', 'production_job_stages.stages_id', 'production_status_features.duration')
					->join('production_status_features', 'production_job_stages.stages_id', '=', 'production_status_features.status_id')->where('production_job_stages.status', 'completed')->whereIn('production_job_stages.job_id', $jobIdArr);

					$completedStageData = $completedStage->get()->toArray();
					$stageCompTimeArr = array_column($completedStageData, 'duration');
					$stageCompTime = array_sum($stageCompTimeArr);
					$orderPercentage = ($stageCompTime / $stageTotalTime) * 100;
					$orderInit->where('order_id', $orderId)
					->update([
						'production_percentage' => round($orderPercentage),
					]);
					
					$logData = [
						'job_id' => $stageDataArr['job_id'],
						'title' => 'Job status reverted back with undo',
						'description' => $logDesc,
						'user_type' => !empty($allPostPutVars['user_type']) ? $allPostPutVars['user_type'] : 'admin',
						'user_id' => !empty($allPostPutVars['user_id']) ? $allPostPutVars['user_id'] : 1,
						'created_date' => date_time(
							'today', [], 'string'
						)
					];
					$this->addingProductionLog($logData);
				}
				$jsonResponse = [
					'status' => 1,
					'message' => message('Stage Operation', 'done'),
				];
			}
		}

		return response(
			$response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
		);
	}

}